package oracle_test

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/client"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/foundation/twitchclient"
)

func TestPutV1Event_SuccessMinimum(t *testing.T) {
	a := assert.New(t)

	id := 123
	now := time.Now()
	hawaii, err := time.LoadLocation("US/Hawaii")
	a.NoError(err)

	hawaiiStartTime := time.Date(
		now.Year(),
		now.Month(),
		now.Day(),
		now.Hour(),
		now.Minute(),
		now.Second(),
		0,
		hawaii,
	)
	hawaiiEndTime := hawaiiStartTime.Add(1 * time.Hour)
	tz := "Europe/Kiev"
	title := "Just your normal title"
	gameID := 999

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		a.Equal(fmt.Sprintf("/v1/events/%d", id), r.URL.EscapedPath())
		a.Equal(http.MethodPut, r.Method)

		reqBody := &view.PutV1EventInput{}
		err := json.NewDecoder(r.Body).Decode(&reqBody)
		a.NoError(err)
		a.NotNil(reqBody)

		a.Equal(hawaiiStartTime.UTC(), reqBody.StartTimeUTC)
		a.Equal(hawaiiEndTime.UTC(), reqBody.EndTimeUTC)
		a.Equal(tz, reqBody.TimeZoneID)
		a.Equal(title, reqBody.Title)
		a.Nil(reqBody.Description)
		a.Equal(gameID, reqBody.GameID)

		payload := `
			{
				"status": 200,
				"message": "success!",
				"meta": {},
				"data": {}
			}
		`

		_, err = w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	payload := &view.PutV1EventInput{
		StartTimeUTC: hawaiiStartTime.UTC(),
		EndTimeUTC:   hawaiiEndTime.UTC(),
		TimeZoneID:   tz,
		Title:        title,
		GameID:       gameID,
	}

	resp, err := client.PutV1Event(context.Background(), id, payload, nil)
	a.NoError(err)

	if a.NotNil(resp) {
		a.Equal(http.StatusOK, resp.Status)
	}
}

func TestPutV1Event_SuccessAllFields(t *testing.T) {
	a := assert.New(t)

	id := 123
	channelID := 2000
	now := time.Now()

	hawaii, err := time.LoadLocation("US/Hawaii")
	a.NoError(err)

	hawaiiStartTime := time.Date(
		now.Year(),
		now.Month(),
		now.Day(),
		now.Hour(),
		now.Minute(),
		now.Second(),
		0,
		hawaii,
	)

	hawaiiEndTime := hawaiiStartTime.Add(1 * time.Hour)
	tz := "Europe/Kiev"
	title := "Just your normal title"
	description := "a regular boring description"
	gameID := 1234
	coverImageID := "someid"
	coverImageSourceURL := "http://test.test"
	coverImageURLTemplate := "http://test.test-TEMPLATE"

	createdAt, err := time.Parse(time.RFC3339, "2017-02-05T01:50:58Z")
	a.NoError(err)
	updatedAt, err := time.Parse(time.RFC3339, "2017-03-05T01:50:58Z")
	a.NoError(err)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		a.Equal(fmt.Sprintf("/v1/events/%d", id), r.URL.EscapedPath())
		a.Equal(http.MethodPut, r.Method)

		reqBody := &view.PutV1EventInput{}
		err := json.NewDecoder(r.Body).Decode(&reqBody)
		a.NoError(err)

		a.Equal(hawaiiStartTime.UTC(), reqBody.StartTimeUTC)
		a.Equal(hawaiiEndTime.UTC(), reqBody.EndTimeUTC)
		a.Equal(tz, reqBody.TimeZoneID)
		a.Equal(title, reqBody.Title)
		a.Equal(description, *reqBody.Description)
		a.Equal(gameID, reqBody.GameID)
		a.Equal(coverImageID, *reqBody.CoverImageID)

		payloadFmt := `
			{
				"status": 200,
				"message": "success!",
				"meta": {
					"status": "available"
				},
				"data": {
					"id": %d,
					"channel_id": %d,
					"start_time_utc": "%s",
					"end_time_utc": "%s",
					"time_zone_id": "%s",
					"title": "%s",
					"description": "%s",
					"game_id": %d,
					"created_at_utc": "%s",
					"updated_at_utc": "%s",
					"cover_image_source_url": "%s",
					"cover_image_url_template": "%s"
				}
			}
		`

		payload := fmt.Sprintf(
			payloadFmt,
			id,
			channelID,
			hawaiiStartTime.UTC().Format(time.RFC3339),
			hawaiiEndTime.UTC().Format(time.RFC3339),
			tz,
			title,
			description,
			gameID,
			createdAt.Format(time.RFC3339),
			updatedAt.Format(time.RFC3339),
			coverImageSourceURL,
			coverImageURLTemplate,
		)

		_, err = w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	payload := &view.PutV1EventInput{
		Title:        title,
		StartTimeUTC: hawaiiStartTime.UTC(),
		EndTimeUTC:   hawaiiEndTime.UTC(),
		TimeZoneID:   tz,
		Description:  &description,
		GameID:       gameID,
		CoverImageID: &coverImageID,
	}

	resp, err := client.PutV1Event(context.Background(), id, payload, nil)
	a.NoError(err)

	if a.NotNil(resp) {
		a.Equal(http.StatusOK, resp.Status)
		a.Equal("success!", resp.Message)
		a.Equal(view.EventStatusAvailable, resp.Meta.Status)
		a.Equal(id, resp.Data.ID)
		a.Equal(channelID, resp.Data.ChannelID)
		a.Equal(hawaiiStartTime.UTC(), resp.Data.StartTimeUTC)
		a.Equal(hawaiiEndTime.UTC(), resp.Data.EndTimeUTC)
		a.Equal(tz, resp.Data.TimeZoneID)
		a.Equal(title, resp.Data.Title)

		if a.NotNil(resp.Data.Description) {
			a.Equal(description, *resp.Data.Description)
		}

		a.Equal(gameID, resp.Data.GameID)
		a.Equal(createdAt, resp.Data.CreatedAtUTC)
		a.Equal(updatedAt, *resp.Data.UpdatedAtUTC)

		if a.NotNil(resp.Data.CoverImageSourceURL) {
			a.Equal(coverImageSourceURL, *resp.Data.CoverImageSourceURL)
		}

		if a.NotNil(resp.Data.CoverImageURLTemplate) {
			a.Equal(coverImageURLTemplate, *resp.Data.CoverImageURLTemplate)
		}
	}
}
