package oracle_test

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/client"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/foundation/twitchclient"
)

func TestPutV1UserEventNotificationSettings_Failure(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusBadRequest)

		payload := `
			{
				"status": 400,
				"error": "Invalid user ID"
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	input := &view.PutV1UserEventNotificationSettingsInput{}

	resp, err := client.PutV1UserEventNotificationSettings(context.Background(), 0, 0, 0, input, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestPutV1UserEventNotificationSettings_MalformedResponse(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "malformed JSON string with an extra double-quote""
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	input := &view.PutV1UserEventNotificationSettingsInput{}

	resp, err := client.PutV1UserEventNotificationSettings(context.Background(), 0, 0, 0, input, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestPutV1UserEventNotificationSettings_DefaultFalseValue(t *testing.T) {
	a := assert.New(t)

	userID := 123
	eventID := 1

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		expectedPath := fmt.Sprintf("/v1/users/%d/notifications/events/%d/settings", userID, eventID)

		a.Equal(expectedPath, r.URL.EscapedPath())
		a.Equal(http.MethodPut, r.Method)

		reqBody := &view.PutV1UserEventNotificationSettingsInput{}
		err := json.NewDecoder(r.Body).Decode(&reqBody)
		a.NoError(err)

		a.False(reqBody.EmailEnabled)

		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "",
				"data": {
					"email_enabled": false
				}
			}
		`

		_, err = w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	input := &view.PutV1UserEventNotificationSettingsInput{}

	resp, err := client.PutV1UserEventNotificationSettings(context.Background(), userID, eventID, userID, input, nil)
	a.NoError(err)

	if a.NotNil(resp) {
		a.Equal(http.StatusOK, resp.Status)
		a.Empty(resp.Message)
		a.False(resp.Data.EmailEnabled)
	}
}

func TestPutV1UserEventNotificationSettings_Success(t *testing.T) {
	a := assert.New(t)

	userID := 123
	eventID := 1

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		expectedPath := fmt.Sprintf("/v1/users/%d/notifications/events/%d/settings", userID, eventID)

		a.Equal(expectedPath, r.URL.EscapedPath())
		a.Equal(http.MethodPut, r.Method)

		reqBody := &view.PutV1UserEventNotificationSettingsInput{}
		err := json.NewDecoder(r.Body).Decode(&reqBody)
		a.NoError(err)

		a.True(reqBody.EmailEnabled)

		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "",
				"data": {
					"email_enabled": true
				}
			}
		`

		_, err = w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	input := &view.PutV1UserEventNotificationSettingsInput{
		EmailEnabled: true,
	}

	resp, err := client.PutV1UserEventNotificationSettings(context.Background(), userID, eventID, userID, input, nil)
	a.NoError(err)

	if a.NotNil(resp) {
		a.Equal(http.StatusOK, resp.Status)
		a.Empty(resp.Message)
		a.True(resp.Data.EmailEnabled)
	}
}
