package config

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"runtime"
	"strings"

	log "github.com/Sirupsen/logrus"
	"github.com/heroku/rollrus"
)

const (
	development = "development"
	staging     = "staging"
	production  = "production"

	devConfigAndSecretsPath = "development.json"
	configPath              = "/var/app/current/config.json"

	// The application secrets JSON path is defined in the Elastic Beanstalk
	// extensions for the Sandstorm-Agent:
	//
	// .ebextensions/data/sandstorm-agent/conf.d/04-secrets.conf
	// .ebextensions/data/sandstorm-agent/templates.d/secrets
	secretsPath = "/var/app/secrets.json"

	// The Rollbar token path is defined in the Elastic Beanstalk
	// extensions for the Sandstorm-Agent:
	//
	// .ebextensions/data/sandstorm-agent/conf.d/02-rollbar.conf
	// .ebextensions/data/sandstorm-agent/templates.d/rollbar-token
	rollbarTokenPath = "/var/app/rollbar_token"
)

var (
	// Environment is a global variable of the application environment
	// according to the environment variable `ENVIRONMENT`.
	Environment string

	// Values is a global variable containing the fields of all
	// environment-specific configuration values for the application.
	Values values

	// Secrets is a global variable containing the fields of all
	// environment-specific secrets for the application.
	Secrets secrets
)

// values contains the non-secret configuration fields for the application.
//
// After defining a field within the values structure,
// the attribute MUST be defined in the following JSON files
// for decoding during application start-up:
//
// * config/development.json
// * config/staging.json
// * config/production.json
//
// Example:
//
// type values struct {
//   SomeServiceHost string
// }
//
// development.json:
//   {
//     "SomeServiceHost": "http://0.0.0.0"
//   }
type values struct {
	AWSAccountProfile    string
	ApiCacheHostPort     string
	CDNCoverImagePrefix  string
	CronCacheHostPort    string
	DBAddress            string
	DiscoveryServiceHost string
	DynamoDBRegion       string
	DynamoDBTableName    string
	JaxServiceHost       string
	S3CoverImageBucket   string
	S3CoverImageHost     string
	S3Region             string
	SNSTopicArn          string
	StatsHostPort        string
	UsersServiceHost     string
	VinylServiceHost     string
}

// secrets contains the secret configuration values for the application.
//
// For development ONLY, after defining a field in the secrets structure,
// the attribute MUST be defined in config/development.json.
//
// For staging or production, a secret's attribute key and template value
// MUST be added to the following file:
//
// .ebextensions/data/sandstorm-agent/templates.d/secrets
//
// for the Sandstorm-Agent to insert first and then for unmarshaling
// during the start-up of the application.
// The secret MUST also be created within Sandstorm under the same key.
//
// Example:
//
// type secrets struct {
//   SomePassword string
// }
//
// development.json:
//   {
//     "SomePassword": "hunter2"
//   }
//
// .ebextensions/data/sandstorm-agent/templates.d/secrets:
//   {
//     "SomePassword": "{{ key "creator-business/oracle/{ENVIRONMENT}/some_password" }}"
//   }
type secrets struct {
	DBCredentials string
}

func init() {
	Environment = os.Getenv("ENVIRONMENT")
	if Environment != production && Environment != staging {
		Environment = development
	}
}

// SetupRollbarLogging reads the Rollbar access token from file,
// and adds a hook to Logrus for reporting the logs of the preset
// levels to Rollbar.
func SetupRollbarLogging() {
	buf, err := ioutil.ReadFile(rollbarTokenPath)
	if err != nil {
		log.Warn("No Rollbar access token found for ", Environment)
		log.Warn("Logs will not be sent to Rollbar")
		return
	}

	rollrus.SetupLogging(strings.TrimSpace(string(buf)), Environment)
	log.Info("Logs set to send to Rollbar")
}

// Load decodes both non-secret and secret configuration values from JSON.
//
// In development, the `config/development.json` file is loaded to
// populate BOTH the Values and Secrets global variables.
//
// In staging or production, the corresponding non-secret config JSON
// (config/staging.json or config/production.json) is added
// to the /var/app/current directory as config.json during
// the deployment process (See: _scripts/deploy_beanstalk.sh).
//
// The secret config JSON is created by the Sandstorm-Agent
// at /var/app/secrets.json
// (See: .ebextensions/data/sandstorm-agent/conf.d/04-secrets.conf)
func Load() {
	configFilePath := configPath
	secretsFilePath := secretsPath

	if Environment == development {
		_, filename, _, ok := runtime.Caller(0)
		if !ok {
			log.Panic("No caller information to get config file name")
		}

		filePath := fmt.Sprint(path.Dir(filename), "/", devConfigAndSecretsPath)
		configFilePath = filePath
		secretsFilePath = filePath
	}

	Values = loadValues(configFilePath)
	Secrets = loadSecrets(secretsFilePath)
}

func loadValues(filePath string) values {
	v := values{}

	log.Info("Loading config from ", filePath)

	file, err := os.Open(filePath)
	if err != nil {
		log.WithError(err).Panic("Failed to open config JSON")
	}

	err = json.NewDecoder(file).Decode(&v)
	if err != nil {
		log.WithError(err).Panic("Failed to decode config JSON")
	}

	return v
}

func loadSecrets(filePath string) secrets {
	s := secrets{}

	log.Info("Loading secrets from ", filePath)

	file, err := os.Open(filePath)
	if err != nil {
		log.WithError(err).Panic("Failed to open configuration secrets JSON")
	}

	err = json.NewDecoder(file).Decode(&s)
	if err != nil {
		log.WithError(err).Panic("Failed to decode configuration secrets JSON")
	}

	return s
}
