package main

import (
	"os"
	"os/signal"
	"time"

	"code.justin.tv/cb/oracle/config"
	"code.justin.tv/cb/oracle/internal/clients"
	"code.justin.tv/cb/oracle/internal/clients/sns"
	"code.justin.tv/cb/oracle/internal/clients/stats"
	"code.justin.tv/cb/oracle/internal/notifications"
	"code.justin.tv/common/gometrics"
	log "github.com/Sirupsen/logrus"
	_ "github.com/lib/pq"
	"github.com/robfig/cron"
)

func init() {
	config.SetupRollbarLogging()
	config.Load()
}

func main() {
	log.Info("Cron application starting in ", config.Environment)

	// Instantiate client interfaces:
	clientsObj, err := clients.NewClients()

	if err != nil {
		log.WithError(err).Panic("Failed to instantiate clients on application start-up")
	}

	// Stats client
	stats, err := stats.NewStatsClient("oracle-cron")
	if err != nil {
		log.WithError(err).Panic("Failed to instantiate stats client on cron")
	}

	// Gometrics
	gometrics.Monitor(stats, time.Second*5)

	n := &notifications.NotificationHandler{
		Clients: clientsObj,
		Stats:   stats,
	}

	// n.NotifySubscribersForEvents(n.GetEventsAllTime, sns.NotificationType{Email: true}, strings.Replace(time.Now().String(), " ", "", -1))

	// Cron jobs
	c := cron.New()

	// When the oracle event is about to begin -- send onsite and mobile notifications
	//	pushy is smart enough to select between the two depending on if the user is online
	err = c.AddFunc("0 * * * * *", func() {
		n.NotifySubscribersForEvents(n.GetCurrentEvents, sns.NotificationType{Onsite: true, Email: false, Mobile: true}, "current")
	})

	if err != nil {
		log.WithError(err).Panic(`Failed to add "current events" email cron`)
	}

	// An hour before the oracle event is about to begin -- send email notifications
	err = c.AddFunc("0 * * * * *", func() {
		n.NotifySubscribersForEvents(n.GetEventsInNextHour, sns.NotificationType{Email: true, Onsite: false, Mobile: false}, "hour")
	})
	if err != nil {
		log.WithError(err).Panic(`Failed to add "hour before" email cron`)
	}

	// start the cron scheduler in its own goroutine
	c.Start()

	// wait until process is interrupted or killed
	sig := make(chan os.Signal)
	signal.Notify(sig, os.Interrupt, os.Kill)
	<-sig
}
