package api

import (
	"fmt"
	"net/http"
	"strconv"
	"time"

	log "github.com/Sirupsen/logrus"
	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/internal/clients/tracking"
	"code.justin.tv/cb/oracle/view"
)

// V1AdminDeleteEvent sets an event's status to admin_deleted
func (s *Server) V1AdminDeleteEvent(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)
	now := time.Now()

	// Validate URL parameter "event_id":
	eventIDStr := pat.Param(r, "event_id")
	eventID, err := strconv.Atoi(eventIDStr)
	if err != nil || eventID < 0 {
		writer.BadRequest(fmt.Sprintf("Invalid event ID (%s)", eventIDStr))
		return
	}

	// Find existing event record by ID:
	selectedEvent, err := s.DB.SelectEventByID(r.Context(), eventID)
	if err != nil {
		writer.Conflict("Failed to find event record")
		return
	} else if selectedEvent == nil {
		writer.NotFound("Event record not found")
		return
	}

	// first delete image -- if it succeeded then delete event from db
	if selectedEvent.CoverImageUUID != nil {
		channelIDStr := strconv.Itoa(selectedEvent.ChannelID)
		err = s.S3.ExpireSavedImage(r.Context(), channelIDStr, *selectedEvent.CoverImageUUID)
		if err != nil {
			writer.Conflict("Failed to change saved cover image permission to private.")
			return
		}
	}

	// The request orginated from within twitch VPN, we mark this event as
	//	deleted by staff
	_, err = s.DB.UpdateEventStatus(r.Context(), eventID, db.EventStatusAdminDeleted)
	if err != nil {
		writer.Conflict("Failed to update event status")
		return
	}

	// Expire caches
	go func() {
		contextLogger := log.WithFields(log.Fields{
			"method":   r.Method,
			"url":      r.RequestURI,
			"event_id": eventID,
		})

		err = s.Cache.ExpireEventView(eventID)
		if err != nil {
			contextLogger.WithError(err).Warn("api: failed to expire cached event view")
		}

		err = s.Cache.ExpireListEventsView(selectedEvent.ChannelID)
		if err != nil {
			contextLogger.WithError(err).Warn("api: failed to expire cached list events view")
		}
	}()

	// Send tracking data:
	var updatedAtTimestampPtr *float64
	if selectedEvent.UpdatedAtUTC != nil {
		updatedAtTimestamp := float64(selectedEvent.UpdatedAtUTC.Unix())
		updatedAtTimestampPtr = &updatedAtTimestamp
	}

	trackingData := tracking.OracleServerEvent{
		Action:                 tracking.OracleServerAdminDeleteAction,
		ServerTimestamp:        float64(now.Unix()),
		EventID:                selectedEvent.ID,
		ChannelID:              selectedEvent.ChannelID,
		EventStartTimestamp:    float64(selectedEvent.StartTimeUTC.Unix()),
		EventEndTimestamp:      float64(selectedEvent.EndTimeUTC.Unix()),
		Timezone:               selectedEvent.TimeZoneID,
		Title:                  selectedEvent.Title,
		GameID:                 selectedEvent.GameID,
		Description:            selectedEvent.Description,
		Status:                 selectedEvent.Status,
		CreatedAtTimestamp:     float64(selectedEvent.CreatedAtUTC.Unix()),
		UpdatedAtTimestamp:     updatedAtTimestampPtr,
		CoverImageUUID:         selectedEvent.CoverImageUUID,
		FallbackCoverImageUUID: selectedEvent.FallbackCoverImageUUID,
	}

	go s.Tracking.SendServerActionEvent(r.Context(), &trackingData)

	// Format JSON response payload:
	payload := &view.DeleteV1EventOutput{
		Status:  http.StatusOK,
		Message: "Successfully deleted event record.",
	}

	writer.OK(payload)
}
