package api

import (
	"net/http"

	"fmt"
	"time"

	"goji.io"
	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/clients"
	"code.justin.tv/cb/oracle/internal/middleware"
	log "github.com/Sirupsen/logrus"
)

// Server contains the router (goji.Mux) as well as
// the environment Context.
type Server struct {
	*goji.Mux
	*Context
}

// Context holds onto the clients and connections
// of all resources needed for the server.
type Context struct {
	*clients.Clients
}

type statsResponseWriter struct {
	http.ResponseWriter
	status int
}

func NewApiServer(c *clients.Clients) *Server {
	mux := goji.SubMux()
	mux.Use(middleware.PanicRecoverer)
	mux.Use(middleware.AccessLogger)

	return &Server{
		Mux: mux,
		Context: &Context{
			Clients: c,
		},
	}
}

func (s *Server) StatHandleFunc(p *pat.Pattern, h http.HandlerFunc, name string) {
	statHandler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		start := time.Now().UTC()
		sw := &statsResponseWriter{
			ResponseWriter: w,
		}
		h.ServeHTTP(sw, r)

		duration := time.Since(start)

		go func() {
			contextLogger := log.WithFields(log.Fields{
				"stat":   name,
				"status": sw.status,
			})

			statName := fmt.Sprintf("%s.%d", name, sw.status)

			err := s.Stats.Inc(statName, 1, .1)
			if err != nil {
				contextLogger.WithError(err).Error("Failed to increment StatsD count type")
			}

			err = s.Stats.TimingDuration(statName, duration, .1)
			if err != nil {
				contextLogger.WithError(err).Error("Failed to submit StatsD timing type")
			}
		}()
	})

	s.HandleFunc(p, statHandler)
}

func (w *statsResponseWriter) WriteHeader(status int) {
	w.status = status
	w.ResponseWriter.WriteHeader(status)
}
