package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"
	"time"

	log "github.com/Sirupsen/logrus"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/auth"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/internal/clients/tracking"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/common/goauthorization"
)

// V1CreateEvent creates a new event for a user given the PostV1EventInput struct
func (s *Server) V1CreateEvent(w http.ResponseWriter, r *http.Request) {
	s.createEventV1(w, r, true)
}

// V1CreateEvent creates a new event for a user given the PostV1EventInput struct
func (s *Server) createEventV1(w http.ResponseWriter, r *http.Request, validateToken bool) {
	writer := responder.NewResponseWriter(w)
	now := time.Now()

	// Validate JSON request body:
	var reqBody view.PostV1EventInput
	err := json.NewDecoder(r.Body).Decode(&reqBody)
	if err != nil {
		writer.BadRequest("Failed to decode JSON request body")
		return
	}

	err = reqBody.Validate()
	if err != nil {
		writer.BadRequest(err.Error())
		return
	}

	channelIDStr := strconv.Itoa(reqBody.ChannelID)

	if validateToken {
		// Make sure user has permission to edit the channel specified
		capabilities := goauthorization.CapabilityClaims{
			"manage_event": goauthorization.CapabilityClaim{
				"channel_id": channelIDStr,
			},
		}

		err = auth.AuthorizeToken(r, &capabilities)
		if err != nil {
			writer.Forbidden("User does not have permission to edit Event")
			return
		}
	}

	// Validate parameter 'game_id':
	_, err = s.Discovery.GetGameByID(r.Context(), reqBody.GameID)
	if err != nil {
		msg := fmt.Sprintf("Parameter 'game_id' (%d) is not a valid game ID.", reqBody.GameID)
		writer.BadRequest(msg)
		return
	}

	// Attempt to copy image from pending key prefix to saved key prefix
	if reqBody.CoverImageID != nil {
		err = s.Clients.S3.SaveFromPending(r.Context(), channelIDStr, *reqBody.CoverImageID)
		if err != nil {
			writer.BadRequest("Failed to save image.")
			return
		}

		idExists, err := s.DB.ExistingCoverImage(r.Context(), *reqBody.CoverImageID)
		if idExists {
			writer.BadRequest("Image UUID already exists.")
			return
		}

		if err != nil {
			writer.Conflict("Image UUID lookup failed.")
			return
		}
	}

	// Persist new event attributes to database:
	fallbackCoverImageID := db.RandomPresetCoverImageID()

	createParams := &db.Event{
		ChannelID:              reqBody.ChannelID,
		StartTimeUTC:           reqBody.StartTimeUTC.UTC(),
		EndTimeUTC:             reqBody.EndTimeUTC.UTC(),
		TimeZoneID:             reqBody.TimeZoneID,
		Title:                  reqBody.Title,
		GameID:                 reqBody.GameID,
		Description:            reqBody.Description,
		CoverImageUUID:         reqBody.CoverImageID,
		FallbackCoverImageUUID: &fallbackCoverImageID,
	}

	err = s.TMI.CensorEventInformation(r.Context(), createParams)
	if err != nil {
		writer.Conflict("Failed to save event")
		return
	}

	savedEvent, err := s.DB.InsertEvent(r.Context(), createParams)
	if err != nil || savedEvent == nil {
		writer.Conflict("Failed to save event record")
		return
	}

	// Expire cache
	go func() {
		err = s.Cache.ExpireListEventsView(savedEvent.ChannelID)
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"method":   r.Method,
				"url":      r.RequestURI,
				"event_id": savedEvent.ID,
			}).Warn("api: failed to expire cached list events view")
		}
	}()

	// Send tracking data:
	var userPtr *int
	token, err := auth.GetToken(r)
	if err == nil && token != nil {
		userID, err := strconv.Atoi(token.GetSubject())
		if err == nil {
			userPtr = &userID
		}
	}

	trackingData := tracking.OracleServerEvent{
		Action:                 tracking.OracleServerCreateAction,
		ServerTimestamp:        float64(now.Unix()),
		UserID:                 userPtr,
		EventID:                savedEvent.ID,
		ChannelID:              savedEvent.ChannelID,
		EventStartTimestamp:    float64(savedEvent.StartTimeUTC.Unix()),
		EventEndTimestamp:      float64(savedEvent.EndTimeUTC.Unix()),
		Timezone:               savedEvent.TimeZoneID,
		Title:                  savedEvent.Title,
		GameID:                 savedEvent.GameID,
		Description:            savedEvent.Description,
		Status:                 savedEvent.Status,
		CreatedAtTimestamp:     float64(savedEvent.CreatedAtUTC.Unix()),
		CoverImageUUID:         savedEvent.CoverImageUUID,
		FallbackCoverImageUUID: savedEvent.FallbackCoverImageUUID,
	}

	go s.Tracking.SendServerActionEvent(r.Context(), &trackingData)

	// Format JSON response payload:
	payload := &view.PostV1EventOutput{
		Status:  http.StatusOK,
		Message: "Successfully saved event record.",
		Meta: view.PostV1EventOutputMeta{
			Status: savedEvent.Status,
		},
		Data: view.V1EventView{
			ID:                    savedEvent.ID,
			ChannelID:             savedEvent.ChannelID,
			StartTimeUTC:          savedEvent.StartTimeUTC,
			EndTimeUTC:            savedEvent.EndTimeUTC,
			TimeZoneID:            savedEvent.TimeZoneID,
			Title:                 savedEvent.Title,
			Description:           savedEvent.Description,
			GameID:                savedEvent.GameID,
			CreatedAtUTC:          savedEvent.CreatedAtUTC,
			CoverImageID:          savedEvent.CoverImageUUID,
			CoverImageSourceURL:   savedEvent.CoverImageSourceURL(),
			CoverImageURLTemplate: savedEvent.CoverImageURLTemplate(),
		},
	}

	writer.OK(payload)
}
