package api

import (
	"encoding/json"
	"net/http"
	"strconv"

	log "github.com/Sirupsen/logrus"
	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/view"
)

// V1CreateEventExtension creates a new event for a user given the PostV1EventExtentionInput struct.
func (s *Server) V1CreateEventExtension(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)

	eventIDStr := pat.Param(r, "event_id")
	eventID, err := strconv.Atoi(eventIDStr)
	if err != nil {
		writer.BadRequest("Failed to decode event_id")
		return
	}

	// Validate event.
	event, err := s.DB.SelectEventByID(r.Context(), eventID)
	if err != nil {
		writer.NotFound("Event not found")
		return
	}

	// Validate JSON request body:
	var reqBody view.PostV1EventExtensionInput
	err = json.NewDecoder(r.Body).Decode(&reqBody)
	if err != nil {
		writer.BadRequest("Failed to decode JSON request body")
		return
	}

	err = reqBody.Validate()
	if err != nil {
		writer.BadRequest(err.Error())
		return
	}

	createParams := &db.EventExtension{
		EventID: event.ID,
		Key:     reqBody.Key,
		Value:   reqBody.Value,
	}

	savedEvent, err := s.DB.InsertEventExtension(r.Context(), createParams)
	if err != nil || savedEvent == nil {
		writer.Conflict("Failed to save event extension record")
		return
	}

	// Expire cache
	go func() {
		err = s.Cache.ExpireListEventExtensionsView(createParams.EventID)
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"method":   r.Method,
				"url":      r.RequestURI,
				"event_id": savedEvent.EventID,
			}).Warn("api: failed to expire cached list events view")
		}
	}()

	// Format JSON response payload:
	payload := &view.PostV1EventExtensionOutput{
		Status:  http.StatusOK,
		Message: "Successfully saved event extension record.",
	}

	writer.OK(payload)
}
