package api

import (
	"net/http"
	"strconv"

	log "github.com/Sirupsen/logrus"
	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/view"
)

// V1DeleteEventExtension creates a new event for a user given the event_id and key of extension.
func (s *Server) V1DeleteEventExtension(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)

	eventIDStr := pat.Param(r, "event_id")
	eventID, err := strconv.Atoi(eventIDStr)
	if err != nil {
		writer.BadRequest("Failed to decode event_id")
		return
	}

	key := pat.Param(r, "key")

	// Validate event.
	_, err = s.DB.SelectEventByID(r.Context(), eventID)
	if err != nil {
		writer.NotFound("Event not found")
		return
	}

	// Get event extension.
	_, err = s.DB.SelectEventExtensionByEventIDAndKey(r.Context(), eventID, key)
	if err != nil {
		writer.NotFound("Event extension not found")
		return
	}

	err = s.DB.DeleteEventExtensionByEventIDAndKey(r.Context(), eventID, key)
	if err != nil {
		writer.BadRequest("Event extension failed to be deleted")
		return

	}

	// Expire cache
	go func() {
		err = s.Cache.ExpireListEventExtensionsView(eventID)
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"method":   r.Method,
				"url":      r.RequestURI,
				"event_id": eventID,
			}).Warn("api: failed to expire cached list event extensions view")
		}
	}()

	// Format JSON response payload:
	payload := &view.DeleteV1EventExtensionOutput{
		Status:  http.StatusOK,
		Message: "Successfully saved event extension record.",
	}

	writer.OK(payload)
}
