package api

import (
	"errors"

	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/view"
	"golang.org/x/net/context"
)

func (s *Server) buildV1EventView(ctx context.Context, event *db.Event) (*view.V1EventView, error) {
	// Format JSON response payload:
	data := &view.V1EventView{
		ID:                    event.ID,
		ChannelID:             event.ChannelID,
		StartTimeUTC:          event.StartTimeUTC,
		EndTimeUTC:            event.EndTimeUTC,
		TimeZoneID:            event.TimeZoneID,
		Title:                 event.Title,
		Description:           event.Description,
		GameID:                event.GameID,
		CreatedAtUTC:          event.CreatedAtUTC,
		UpdatedAtUTC:          event.UpdatedAtUTC,
		CoverImageID:          event.CoverImageUUID,
		CoverImageSourceURL:   event.CoverImageSourceURL(),
		CoverImageURLTemplate: event.CoverImageURLTemplate(),
	}

	eventExtensions, err := s.DB.SelectEventExtensionsByEventID(ctx, event.ID)
	if err != nil {
		return nil, errors.New("Failed to fetch event extensions")
	}

	// Adding extensions.
	for _, extension := range eventExtensions {
		data.Extensions = append(data.Extensions, &view.V1EventExtensionView{
			EventID: extension.EventID,
			Key:     extension.Key,
			Value:   extension.Value,
		})
	}

	return data, nil
}

func (s *Server) buildV1EventViewList(ctx context.Context, events []*db.Event) ([]*view.V1EventView, error) {

	// Format JSON response payload:
	data := make([]*view.V1EventView, len(events))
	eventIDS := make([]int, len(events))

	for idx, event := range events {
		eventIDS[idx] = event.ID
		data[idx] = &view.V1EventView{
			ID:                    event.ID,
			ChannelID:             event.ChannelID,
			StartTimeUTC:          event.StartTimeUTC,
			EndTimeUTC:            event.EndTimeUTC,
			TimeZoneID:            event.TimeZoneID,
			Title:                 event.Title,
			Description:           event.Description,
			GameID:                event.GameID,
			CreatedAtUTC:          event.CreatedAtUTC,
			UpdatedAtUTC:          event.UpdatedAtUTC,
			CoverImageID:          event.CoverImageUUID,
			CoverImageSourceURL:   event.CoverImageSourceURL(),
			CoverImageURLTemplate: event.CoverImageURLTemplate(),
		}
	}

	eventExtensions, err := s.DB.SelectEventExtensionsByEventIDs(ctx, eventIDS)
	if err != nil {
		return nil, errors.New("Failed to fetch event extensions")
	}

	// Create hash by eventID with []*V1EventExtensionView.
	hashedEventExtensions := make(map[int][]*view.V1EventExtensionView)
	for _, extension := range eventExtensions {
		hashedEventExtensions[extension.EventID] = append(hashedEventExtensions[extension.EventID], &view.V1EventExtensionView{
			EventID: extension.EventID,
			Key:     extension.Key,
			Value:   extension.Value,
		})
	}

	// Add extensions to each event
	for _, eventData := range data {
		eventData.Extensions = hashedEventExtensions[eventData.ID]
	}

	return data, nil
}

func (s *Server) buildV1ManagerEventViewList(ctx context.Context, events []*db.ManagerEvent) ([]*view.V1ManagerEventView, error) {

	// Format JSON response payload:
	data := make([]*view.V1ManagerEventView, len(events))
	eventIDS := make([]int, len(events))

	for idx, event := range events {
		eventIDS[idx] = event.ID
		data[idx] = &view.V1ManagerEventView{
			V1EventView: &view.V1EventView{
				ID:                    event.ID,
				ChannelID:             event.ChannelID,
				StartTimeUTC:          event.StartTimeUTC,
				EndTimeUTC:            event.EndTimeUTC,
				TimeZoneID:            event.TimeZoneID,
				Title:                 event.Title,
				Description:           event.Description,
				GameID:                event.GameID,
				CreatedAtUTC:          event.CreatedAtUTC,
				UpdatedAtUTC:          event.UpdatedAtUTC,
				CoverImageID:          event.CoverImageUUID,
				CoverImageSourceURL:   event.CoverImageSourceURL(),
				CoverImageURLTemplate: event.CoverImageURLTemplate(),
			},
			EmailNotificationCount: event.EnabledUserEmailNotificationCount,
		}
	}

	eventExtensions, err := s.DB.SelectEventExtensionsByEventIDs(ctx, eventIDS)
	if err != nil {
		return nil, errors.New("Failed to fetch event extensions")
	}

	// Create hash by eventID with []*V1EventExtensionView.
	hashedEventExtensions := make(map[int][]*view.V1EventExtensionView)
	for _, extension := range eventExtensions {
		hashedEventExtensions[extension.EventID] = append(hashedEventExtensions[extension.EventID], &view.V1EventExtensionView{
			EventID: extension.EventID,
			Key:     extension.Key,
			Value:   extension.Value,
		})
	}

	// Add extensions to each event
	for _, eventData := range data {
		eventData.Extensions = hashedEventExtensions[eventData.ID]
	}

	return data, nil
}
