package api

import (
	"fmt"
	"net/http"
	"strconv"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/auth"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/common/goauthorization"
)

// V1ListAvailableManagerEvents returns all available events for a given
// channel, ordered by start_time_utc, with optional query parameters:
//
// * end_time_utc_after
// * end_time_utc_before
// * order_by
//
// If no parameters are provided, then we set `end_time_utc_after` to the
// current time (`time.Now()`) and set `order_by` to `asc`
// for current and upcoming events.
//
// V1ListAvailableManagerEvents is different than V1ListAvailableEvents,
// in that the returned events contain additional stats only authorized to the
// event manager.
//
// Additionally, the HTTP views of V1ListAvailableManagerEvents are NOT cached.
func (s *Server) V1ListAvailableManagerEvents(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)

	params, err := ParseV1ListEventsParams(r)
	if err != nil {
		writer.BadRequest(err.Error())
		return
	}

	// Validate client authorization:
	capabilities := goauthorization.CapabilityClaims{
		"manage_event": goauthorization.CapabilityClaim{
			"channel_id": strconv.Itoa(params.ChannelID),
		},
	}

	err = auth.AuthorizeToken(r, &capabilities)
	if err != nil {
		writer.Forbidden("Unauthorized to manage channel's events")
		return
	}

	// Query from database:
	events, err := s.DB.SelectAvailableManagerEventsByChannelID(r.Context(), params)
	if err != nil {
		writer.InternalServerError("Failed to select manager events", err)
		return
	}

	// Building []V1ManagerEventView from Events
	data, err := s.buildV1ManagerEventViewList(r.Context(), events)
	if err != nil {
		writer.NotFound(err.Error())
		return
	}

	payload := &view.GetV1AvailableManagerEventListOutput{
		Status:  http.StatusOK,
		Message: fmt.Sprint("Count of manager events found: ", len(data)),
		Meta: view.GetV1AvailableManagerEventListOutputMeta{
			ChannelID:        params.ChannelID,
			EndTimeUTCAfter:  params.EndTimeAfter,
			EndTimeUTCBefore: params.EndTimeBefore,
			OrderBy:          params.OrderBy,
			Status:           db.EventStatusAvailable,
			Limit:            db.SelectLimit,
		},
		Data: data,
	}

	writer.OK(payload)
}
