package api

import (
	"fmt"
	"net/http"
	"strconv"

	log "github.com/Sirupsen/logrus"
	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/view"
)

// V1ListAvailableEventExtensions returns all available event extensions for a given event.
//
func (s *Server) V1ListEventExtensions(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)

	eventIDStr := pat.Param(r, "event_id")
	eventID, err := strconv.Atoi(eventIDStr)
	if err != nil {
		msg := fmt.Sprintf(responder.InvalidParameter, "event_id", eventIDStr)
		writer.BadRequest(msg)
		return
	}

	// Read from cache
	bytes, err := s.Cache.GetListEventExtensionsView(eventID)
	if err == nil && bytes != nil {
		writer.OKRaw(*bytes)
		return
	}
	// Catch err
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"method":   r.Method,
			"url":      r.RequestURI,
			"event_id": eventID,
		}).Warn("api: failed to cache list event extensions view")
	}

	// Query from database:
	extensions, err := s.DB.SelectEventExtensionsByEventID(r.Context(), eventID)
	if err != nil {
		writer.NotFound("Failed to fetch event extensions")
		return
	}

	// Format JSON response payload:
	data := make([]*view.V1EventExtensionView, len(extensions))
	for idx, extension := range extensions {
		data[idx] = &view.V1EventExtensionView{
			EventID: extension.EventID,
			Key:     extension.Key,
			Value:   extension.Value,
		}
	}

	payload := &view.GetV1EventExtensionListOutput{
		Status:  http.StatusOK,
		Message: fmt.Sprintf("Found %d event extensions", len(data)),
		Data:    data,
	}

	// Cache payload
	go func() {
		err = s.Cache.CacheListEventExtensionsView(eventID, *payload)
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"method":   r.Method,
				"url":      r.RequestURI,
				"event_id": eventID,
				"view":     *payload,
			}).Warn("api: failed to cache list event extensions view")
		}
	}()

	writer.OK(payload)
}
