package api

import (
	"errors"
	"fmt"
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/view"
	log "github.com/Sirupsen/logrus"
)

// V1ListAvailableEvents returns all available events for a given channel,
// ordered by start_time_utc, with optional query parameters:
//
// * end_time_utc_after
// * end_time_utc_before
// * order_by
//
// If no parameters are provided, then we set `end_time_utc_after` to the
// current time (`time.Now()`) and set `order_by` to `asc`
// for current and upcoming events.
func (s *Server) V1ListAvailableEvents(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)

	params, err := ParseV1ListEventsParams(r)
	if err != nil {
		writer.BadRequest(err.Error())
		return
	}

	// Read from cache
	bytes, err := s.Cache.GetListEventsView(*params)
	if err == nil && bytes != nil {
		writer.OKRaw(*bytes)
		return
	}
	// Catch err
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"method": r.Method,
			"url":    r.RequestURI,
			"params": *params,
		}).Warn("api: failed to cache list events view")
	}

	// Query from database:
	events, err := s.DB.SelectAvailableEventsByChannelID(r.Context(), params)
	if err != nil {
		writer.NotFound("Failed to fetch events")
		return
	}

	// Building []V1EventView from Events
	data, err := s.buildV1EventViewList(r.Context(), events)
	if err != nil {
		writer.NotFound(err.Error())
		return
	}

	payload := &view.GetV1AvailableEventListOutput{
		Status:  http.StatusOK,
		Message: fmt.Sprintf("Found %d events", len(data)),
		Meta: view.GetV1AvailableEventListOutputMeta{
			ChannelID:        params.ChannelID,
			EndTimeUTCAfter:  params.EndTimeAfter,
			EndTimeUTCBefore: params.EndTimeBefore,
			OrderBy:          params.OrderBy,
			Status:           db.EventStatusAvailable,
			Limit:            db.SelectLimit,
		},
		Data: data,
	}

	// Cache payload
	go func() {
		err = s.Cache.CacheListEventsView(*params, *payload)
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"method": r.Method,
				"url":    r.RequestURI,
				"params": *params,
				"view":   *payload,
			}).Warn("api: failed to cache list events view")
		}
	}()

	writer.OK(payload)
}

func parseTimeParam(timeStr string, defaultTime time.Time) (time.Time, error) {
	if len(timeStr) == 0 {
		return defaultTime, nil
	}
	t, err := time.Parse(time.RFC3339, timeStr)
	if err != nil {
		return defaultTime, err
	}

	return t.UTC(), nil
}

func ParseV1ListEventsParams(r *http.Request) (*db.EventListParams, error) {
	channelIDStr := r.URL.Query().Get("channel_id")
	channelID, err := strconv.Atoi(channelIDStr)
	if err != nil {
		msg := fmt.Sprintf(responder.InvalidParameter, "channel_id", channelIDStr)
		return nil, errors.New(msg)
	}

	timeZero := time.Unix(0, 0)
	timeInf := time.Now().AddDate(1000, 0, 0) // one thousand years from now

	// validate time bounds
	endTimeBeforeStr := r.URL.Query().Get("end_time_utc_before")
	endTimeBefore, err := parseTimeParam(endTimeBeforeStr, timeInf)
	if err != nil {
		msg := fmt.Sprint(responder.InvalidTimeFormat, "end_time_utc_before")
		return nil, errors.New(msg)
	}

	endTimeAfterStr := r.URL.Query().Get("end_time_utc_after")
	endTimeAfter, err := parseTimeParam(endTimeAfterStr, timeZero)
	if err != nil {
		msg := fmt.Sprint(responder.InvalidTimeFormat, "end_time_utc_after")
		return nil, errors.New(msg)
	}

	// default to finding all current and upcoming events if no parameters given
	if len(endTimeBeforeStr) == 0 && len(endTimeAfterStr) == 0 {
		endTimeAfter = time.Now()
	}

	orderBy := r.URL.Query().Get("order_by")
	// default to order by = ASC
	if len(orderBy) == 0 || (orderBy != "asc" && orderBy != "desc") {
		orderBy = "asc"
	}

	eventListQuery := &db.EventListParams{
		ChannelID:     channelID,
		EndTimeBefore: endTimeBefore,
		EndTimeAfter:  endTimeAfter,
		OrderBy:       orderBy,
	}

	return eventListQuery, nil
}
