package api

import (
	"fmt"
	"net/http"
	"strconv"

	log "github.com/Sirupsen/logrus"
	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/view"
)

// V1ReadEvent fetches an existing event record given the event's ID.
func (s *Server) V1ReadEvent(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)

	// Validate URL parameter "event_id":
	eventIDStr := pat.Param(r, "event_id")
	eventID, err := strconv.Atoi(eventIDStr)

	if err != nil || eventID <= 0 {
		writer.BadRequest(fmt.Sprintf("Invalid event ID (%s)", eventIDStr))
		return
	}

	// Try to read from cache
	response, err := s.Cache.GetEventView(eventID)
	if err == nil && response != nil {
		writer.OKRaw(*response)
		return
	}

	// Query for the event record from the database:
	selectedEvent, err := s.DB.SelectEventByID(r.Context(), eventID)
	if err != nil {
		writer.InternalServerError("Failed to query event record", err)
		return
	} else if selectedEvent == nil {
		writer.NotFound("Event record not found")
		return
	}

	convertedEvent, err := s.buildV1EventView(r.Context(), selectedEvent)
	if err != nil {
		writer.InternalServerError("Failed to hydrate event record", err)
		return
	}

	// Format JSON response payload:
	payload := &view.GetV1EventOutput{
		Status: http.StatusOK,
		Meta: view.GetV1EventOutputMeta{
			Status: selectedEvent.Status,
		},
		Data: *convertedEvent,
	}

	// Cache payload
	go func() {
		err = s.Cache.CacheEventView(eventID, *payload)
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"method": r.Method,
				"url":    r.RequestURI,
				"view":   *payload,
			}).Warn("api: failed to cache event view")
		}
	}()

	writer.OK(payload)
}
