package api

import (
	"fmt"
	"net/http"
	"strconv"

	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/auth"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/internal/clients/dynamodb"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/common/goauthorization"
)

// V1ReadUserEventNotificationSettings fetches a event_id:user_id pair from
// Dynamo for the user's event notification settings.
func (s *Server) V1ReadUserEventNotificationSettings(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)

	// Validate GET request parameters:
	eventIDStr := pat.Param(r, "event_id")
	eventID, err := strconv.Atoi(eventIDStr)
	if err != nil || eventID <= 0 {
		writer.BadRequest(fmt.Sprintf("Invalid event ID (%s)", eventIDStr))
		return
	}

	userIDStr := pat.Param(r, "user_id")
	userID, err := strconv.Atoi(userIDStr)
	if err != nil || userID <= 0 {
		writer.BadRequest(fmt.Sprintf("Invalid user ID (%s)", userIDStr))
		return
	}

	// Validate the user's permission to read the event notification settings:
	requestingUserIDStr := r.URL.Query().Get("user_id")
	if requestingUserIDStr != "" {
		if requestingUserIDStr != userIDStr {
			writer.Forbidden("User does not have permission to get notification")
			return
		}
	} else {
		// Fall back to cartman tokens
		capabilities := goauthorization.CapabilityClaims{
			"manage_user_event_notification_settings": goauthorization.CapabilityClaim{
				"user_id": userIDStr,
			},
		}

		err = auth.AuthorizeToken(r, &capabilities)
		if err != nil {
			writer.Forbidden("User does not have permission to get notification")
			return
		}
	}

	// Find existing event record by ID:
	selectedEvent, err := s.DB.SelectEventByID(r.Context(), eventID)
	if err != nil || selectedEvent == nil {
		msg := fmt.Sprintf("Event record (ID %d) not found", eventID)
		writer.NotFound(msg)
		return
	} else if selectedEvent.Status != db.EventStatusAvailable {
		msg := fmt.Sprintf("Event (ID %d) not available for notification", eventID)
		writer.BadRequest(msg)
		return
	}

	// Query from DynamoDB:
	notif, err := s.DynamoDB.GetEventNotificationsForUser(r.Context(), eventID, userIDStr)
	if err == dynamodb.ErrNoQueryItems {
		// Default to false if the user has not set the email notification setting.
		writer.OK(&view.GetV1UserEventNotificationSettingsOutput{
			Status: http.StatusOK,
			Data: view.GetV1UserEventNotificationSettingsOutputData{
				EmailEnabled: false,
			},
		})

		return
	}

	if err != nil {
		msg := fmt.Sprintf("Failed to get notification (event ID %d, user ID %d)", eventID, userID)
		writer.InternalServerError(msg, err)
		return
	}

	// Format GET response payload:
	payload := &view.GetV1UserEventNotificationSettingsOutput{
		Status: http.StatusOK,
		Data: view.GetV1UserEventNotificationSettingsOutputData{
			EmailEnabled: notif.Email,
		},
	}

	writer.OK(payload)
}
