package api

import (
	"fmt"
	"net/http"
	"strconv"
	"strings"

	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/view"
)

// V1ReadUserEventNotificationSettings fetches a event_id:user_id pair from
// Dynamo for the user's event notification settings.
func (s *Server) V1ReadUserEventsNotificationSettings(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)

	// Validate GET request parameters:
	eventIDStrs := strings.Split(r.URL.Query().Get("event_ids"), ",")
	eventIDs := make([]int, 0, len(eventIDStrs))
	for _, eventIDStr := range eventIDStrs {
		eventID, err := strconv.Atoi(eventIDStr)
		if err != nil || eventID <= 0 {
			writer.BadRequest(fmt.Sprintf("Invalid event ID (%s)", eventIDStr))
			return
		}
		eventIDs = append(eventIDs, eventID)
	}

	requestingUserIDStr := r.URL.Query().Get("user_id")
	requestingUserID, err := strconv.Atoi(requestingUserIDStr)
	if err != nil || requestingUserID <= 0 {
		writer.BadRequest(fmt.Sprintf("Invalid requesting user ID (%s)", requestingUserIDStr))
		return
	}

	userIDStr := pat.Param(r, "user_id")
	userID, err := strconv.Atoi(userIDStr)
	if err != nil || userID <= 0 {
		writer.BadRequest(fmt.Sprintf("Invalid user ID (%s)", userIDStr))
		return
	}

	if requestingUserID != userID {
		writer.Forbidden("User does not have permission to get notification")
		return
	}

	// Query from DynamoDB:
	items, err := s.DynamoDB.GetEventsNotificationsForUser(r.Context(), eventIDs, userIDStr)
	if err != nil {
		msg := fmt.Sprintf("Failed to get notification (event IDs %s, user ID %d)", eventIDStrs, userID)
		writer.InternalServerError(msg, err)
		return
	}

	// Format GET response payload:
	data := make([]view.GetV1UserEventsNotificationSettingsOutputData, 0, len(items))
	for _, item := range items {
		data = append(data, view.GetV1UserEventsNotificationSettingsOutputData{
			EventID:      item.EventID,
			EmailEnabled: item.Email,
		})
	}

	payload := &view.GetV1UserEventsNotificationSettingsOutput{
		Status: http.StatusOK,
		Data:   data,
	}

	writer.OK(payload)
}
