package responder

import (
	"encoding/json"
	"fmt"
	"net/http"

	log "github.com/Sirupsen/logrus"
)

const (
	ErrorDecodingJSON string = "Failed to decode JSON request body"
	InvalidChannelID  string = "Parameter 'channel_id' (%v) is not a valid channel ID."
	InvalidGameID     string = "Parameter 'game_id' (%v) is not a valid game ID."
	InvalidTimeFormat string = "Paramter '%s' must be in RFC 3339 format."
	InvalidParameter  string = "Paramter '%v' (%v) is not in a valid format."
)

// ResponseWriter is used for all HTTP responses to the client.
// (Copied over from cb/dashy)
type ResponseWriter interface {
	// 2xx responses
	OK(interface{})
	OKRaw(bytes []byte)

	// 4xx responses
	BadRequest(interface{})
	Unauthorized(interface{})
	Forbidden(interface{})
	NotFound(interface{})
	Conflict(interface{})
	Gone(interface{})
	UnprocessableEntity(interface{})

	// 5xx responses
	InternalServerError(interface{}, error)
	ServiceUnavailable(interface{}, error)
}

type responseWriter struct {
	http.ResponseWriter
}

// NewResponseWriter returns a ResponseWriter for the given request.
func NewResponseWriter(w http.ResponseWriter) ResponseWriter {
	return &responseWriter{w}
}

// OK returns a 200 status code and serializes the given structure as JSON.
func (w *responseWriter) OK(data interface{}) {
	w.serveJSON(http.StatusOK, data)
}

// OK returns a 200 status code and writes raw bytes into response.
func (w *responseWriter) OKRaw(bytes []byte) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(http.StatusOK)
	_, err := w.Write(bytes)
	if err != nil {
		w.InternalServerError("OKRaw failed", err)
	}
}

// BadRequest returns a 400 status code.
func (w *responseWriter) BadRequest(message interface{}) {
	w.serveError(http.StatusBadRequest, message)
}

// Unauthorized returns a 401 status code.
func (w *responseWriter) Unauthorized(message interface{}) {
	w.serveError(http.StatusUnauthorized, message)
}

// Forbidden returns a 403 status code.
func (w *responseWriter) Forbidden(message interface{}) {
	w.serveError(http.StatusForbidden, message)
}

// NotFound returns a 404 status code.
func (w *responseWriter) NotFound(message interface{}) {
	w.serveError(http.StatusNotFound, message)
}

// Conflict returns a 409 status code.
func (w *responseWriter) Conflict(message interface{}) {
	w.serveError(http.StatusConflict, message)
}

// Gone returns a 410 status code.
func (w *responseWriter) Gone(message interface{}) {
	w.serveError(http.StatusGone, message)
}

// UnprocessibleEntity returns a 422 status code.
func (w *responseWriter) UnprocessableEntity(message interface{}) {
	w.serveError(http.StatusUnprocessableEntity, message)
}

// InternalServerError returns a 500 status code and logs the error.
func (w *responseWriter) InternalServerError(message interface{}, err error) {
	status := http.StatusInternalServerError

	log.WithError(err).Error(http.StatusText(status))
	w.serveError(status, message)
}

// ServiceUnavailable returns a 503 status code and logs the error.
func (w *responseWriter) ServiceUnavailable(message interface{}, err error) {
	status := http.StatusServiceUnavailable

	log.WithError(err).Error(http.StatusText(status))
	w.serveError(status, message)
}

func (w *responseWriter) serveJSON(status int, data interface{}) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)

	err := json.NewEncoder(w).Encode(data)
	if err != nil {
		status = http.StatusInternalServerError

		log.WithError(err).Error("Failed to write HTTP JSON response")
		http.Error(w, http.StatusText(status), status)
	}
}

func (w *responseWriter) serveError(status int, message interface{}) {
	if message != nil {
		message = fmt.Sprintf("%s", message)
	}

	response := map[string]interface{}{
		"error":   http.StatusText(status),
		"status":  status,
		"message": message,
	}

	w.serveJSON(status, response)
}
