package api

import (
	"context"
	"fmt"
	"net/http"
	"sort"
	"strconv"
	"time"

	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/view"
	log "github.com/Sirupsen/logrus"
)

const (
	GameEventsTimeRange = 2 * 7 * 24 * time.Hour
	VodExtensionKey     = "vod"

	concurrentMinimum     = 100
	eventQueryLimit       = 100
	futureSuggestionLimit = 5
	liveSuggestionLimit   = 4
	pastSuggestionLimit   = 5
	totalSuggestionLimit  = 7
)

// V1SuggestEventsForGame provides a selection of current, future, and past events for a game
func (s *Server) V1SuggestEventsForGame(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)
	now := time.Now().UTC()

	gameIDStr := pat.Param(r, "game_id")
	gameID, err := strconv.Atoi(gameIDStr)
	if err != nil {
		msg := fmt.Sprintf(responder.InvalidParameter, "game_id", gameIDStr)
		writer.BadRequest(msg)
		return
	}

	// Try to serve cache.
	// Read from cache
	bytes, err := s.Cache.GetGameEventsView(gameID)
	if err == nil && bytes != nil {
		writer.OKRaw(*bytes)
		return
	}
	// Catch err
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"method":  r.Method,
			"url":     r.RequestURI,
			"game_id": gameID,
		}).Warn("api: failed to load cache for game events")
	}

	// Load all suggested events.
	ctx := r.Context()

	liveLimit := eventQueryLimit
	liveEventsResp, err := s.DB.SelectLiveEventsByGameForTime(ctx, now, gameID, &db.EventSelectionParams{
		Limit: &liveLimit,
	})
	liveEvents := []*db.Event{}
	if err != nil {
		// Log and proceed
		log.WithError(err).Warn("api: failed to fetch live events")
	} else {
		liveEvents = liveEventsResp.Events
	}
	liveSuggestions, err := s.FilterAndSortLiveEvents(ctx, liveEvents, liveSuggestionLimit, gameID)
	if err != nil {
		log.WithError(err).Warn("api: failed to get additional info about live events")
	}

	// Reach 2 weeks into the future
	futureWindow := now.Add(GameEventsTimeRange)
	futureLimit := totalSuggestionLimit - len(liveSuggestions)
	if futureLimit > futureSuggestionLimit {
		futureLimit = futureSuggestionLimit
	}
	futureEventsResp, err := s.DB.SelectGameEventsWithinTimeRange(ctx, now, futureWindow, gameID, &db.EventSelectionParams{
		Limit: &futureLimit,
	})
	futureEvents := []*db.Event{}
	if err != nil {
		// Log and proceed
		log.WithError(err).Warn("api: failed to fetch future events")
	} else {
		futureEvents = futureEventsResp.Events
	}

	futureSuggestions, err := s.FilterAndSortFutureEvents(ctx, futureEvents, len(futureEvents))
	if err != nil {
		writer.InternalServerError(err.Error(), err)
		return
	}

	// Reach 2 weeks into the past
	pastLimit := pastSuggestionLimit
	pastWindow := now.Add(-GameEventsTimeRange)
	pastEventsResp, err := s.DB.SelectGameEventsWithinTimeRange(
		ctx,
		pastWindow,
		now,
		gameID,
		&db.EventSelectionParams{
			Limit: &pastLimit,
		},
	)
	pastEvents := []*db.Event{}
	if err != nil {
		// Log and proceed
		log.WithError(err).Warn("api: failed to fetch past events")
	} else {
		pastEvents = pastEventsResp.Events
	}

	pastSuggestions, err := s.FilterAndSortPastEvents(ctx, pastEvents, pastSuggestionLimit)
	if err != nil {
		writer.InternalServerError(err.Error(), err)
		return
	}

	// Payload
	payload := &view.GetV1EventSuggestionsOutput{
		Status:  http.StatusOK,
		Message: "Found suggestions for game",
		Meta: view.GetV1EventSuggestionsOutputMeta{
			Status: db.EventStatusAvailable,
		},
		Data: view.GetV1EventSuggestionsOutputData{
			Live:   liveSuggestions,
			Past:   pastSuggestions,
			Future: futureSuggestions,
		},
	}

	// Cache payload
	go func() {
		err = s.Cache.CacheGameEventsView(gameID, *payload)
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"method":  r.Method,
				"url":     r.RequestURI,
				"game_id": gameID,
				"view":    *payload,
			}).Warn("api: failed to cache list events view")
		}
	}()

	writer.OK(payload)
}

type liveWithConcurrent struct {
	*view.V1EventView
	ConcurrentViewers int
}

type eventsByStartTime []*view.V1EventView

func (e eventsByStartTime) Len() int           { return len(e) }
func (e eventsByStartTime) Swap(i, j int)      { e[i], e[j] = e[j], e[i] }
func (e eventsByStartTime) Less(i, j int) bool { return e[i].StartTimeUTC.Before(e[j].StartTimeUTC) }

type liveByConcurrent []*liveWithConcurrent

func (l liveByConcurrent) Len() int      { return len(l) }
func (l liveByConcurrent) Swap(i, j int) { l[i], l[j] = l[j], l[i] }
func (l liveByConcurrent) Less(i, j int) bool {
	if l[j] == nil {
		return l[i] != nil
	}
	if l[i] == nil {
		return false
	}
	return l[i].ConcurrentViewers > l[j].ConcurrentViewers
}

func (s *Server) FilterAndSortFutureEvents(ctx context.Context, futureEvents []*db.Event, limit int) ([]*view.V1EventView, error) {
	futureSuggestions, err := s.buildV1EventViewList(ctx, futureEvents)
	if err != nil {
		return nil, err
	}
	sort.Sort(eventsByStartTime(futureSuggestions))
	return futureSuggestions[:limit], nil
}

func (s *Server) FilterAndSortLiveEvents(ctx context.Context, liveEvents []*db.Event, limit int, gameID int) ([]*view.V1EventView, error) {
	liveChannelIDs := make([]string, len(liveEvents))
	eventsByChannelID := make(map[int]*db.Event)
	for idx, event := range liveEvents {
		liveChannelIDs[idx] = strconv.Itoa(event.ChannelID)
		eventsByChannelID[event.ChannelID] = event
	}
	liveStreams, err := s.Jax.GetStreamsByChannelIDs(ctx, liveChannelIDs, nil, nil)
	if err != nil {
		return nil, err
	}

	liveWithConcurrents := make(liveByConcurrent, 0, len(liveStreams.Hits))
	for _, stream := range liveStreams.Hits {
		concurrent := 0
		if stream.Properties.Usher.ChannelCount != nil {
			concurrent = *stream.Properties.Usher.ChannelCount
		}
		channelID := *stream.Properties.Rails.ChannelID

		data, err := s.buildV1EventView(ctx, eventsByChannelID[channelID])
		if err != nil {
			return nil, err
		}
		if concurrent >= concurrentMinimum && stream.Properties.Rails.GameID != nil && *stream.Properties.Rails.GameID == gameID {
			liveWithConcurrents = append(liveWithConcurrents, &liveWithConcurrent{
				ConcurrentViewers: concurrent,
				V1EventView:       data,
			})
		}
	}

	sort.Sort(liveWithConcurrents)
	selectionLimit := limit
	if selectionLimit > len(liveWithConcurrents) {
		selectionLimit = len(liveWithConcurrents)
	}

	res := make([]*view.V1EventView, selectionLimit)
	for idx, wrappedEvent := range liveWithConcurrents[:selectionLimit] {
		res[idx] = wrappedEvent.V1EventView
	}
	return res, nil
}

type pastWithViewCount struct {
	*view.V1EventView
	VODViews int64
}
type pastByViewCount []*pastWithViewCount

func (p pastByViewCount) Len() int           { return len(p) }
func (p pastByViewCount) Swap(i, j int)      { p[i], p[j] = p[j], p[i] }
func (p pastByViewCount) Less(i, j int) bool { return p[i].VODViews < p[j].VODViews }

func (s *Server) FilterAndSortPastEvents(ctx context.Context, pastEvents []*db.Event, limit int) ([]*view.V1EventView, error) {
	pastSuggestions, err := s.buildV1EventViewList(ctx, pastEvents)
	if err != nil {
		return nil, err
	}
	sort.Sort(sort.Reverse(eventsByStartTime(pastSuggestions)))
	if len(pastSuggestions) > limit {
		return pastSuggestions[:limit], nil
	}
	return pastSuggestions, nil
}
