package api_test

import (
	"context"
	"testing"
	"time"

	"code.justin.tv/cb/oracle/internal/api"
	"code.justin.tv/cb/oracle/internal/clients"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/mocks/clients"
	"code.justin.tv/web/jax/client"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

var EventA = db.Event{
	ID:           123,
	ChannelID:    321,
	StartTimeUTC: time.Now().Add(-1 * time.Hour).UTC(),
	EndTimeUTC:   time.Now().Add(5 * time.Hour).UTC(),
	TimeZoneID:   "America/Los_Angeles",
	Title:        "This is a live event",
	GameID:       488552,
	Description:  nil,
	CreatedAtUTC: time.Now().Add(-1 * time.Hour).UTC(),
}

var EventB = db.Event{
	ID:           234,
	ChannelID:    432,
	StartTimeUTC: time.Now().Add(-2 * time.Hour).UTC(),
	EndTimeUTC:   time.Now().Add(4 * time.Hour).UTC(),
	TimeZoneID:   "America/Los_Angeles",
	Title:        "This is a another live event",
	GameID:       488552,
	Description:  nil,
	CreatedAtUTC: time.Now().Add(-1 * time.Hour).UTC(),
}

var EventC = db.Event{
	ID:           345,
	ChannelID:    543,
	StartTimeUTC: time.Now().Add(-90 * time.Minute).UTC(),
	EndTimeUTC:   time.Now().Add(40 * time.Minute).UTC(),
	TimeZoneID:   "America/Los_Angeles",
	Title:        "This is yet another live event",
	GameID:       488552,
	Description:  nil,
	CreatedAtUTC: time.Now().Add(-1 * time.Hour).UTC(),
}

var EventD = db.Event{
	ID:           456,
	ChannelID:    654,
	StartTimeUTC: time.Now().Add(-4 * time.Hour).UTC(),
	EndTimeUTC:   time.Now().Add(30 * time.Minute).UTC(),
	TimeZoneID:   "America/Los_Angeles",
	Title:        "This is the last live event",
	GameID:       488552,
	Description:  nil,
	CreatedAtUTC: time.Now().Add(-1 * time.Hour).UTC(),
}

var EventE = db.Event{
	ID:           567,
	ChannelID:    765,
	StartTimeUTC: time.Now().Add(-4 * time.Hour).UTC(),
	EndTimeUTC:   time.Now().Add(30 * time.Minute).UTC(),
	TimeZoneID:   "America/Los_Angeles",
	Title:        "This is the same time as EventD",
	GameID:       488552,
	Description:  nil,
	CreatedAtUTC: time.Now().Add(-1 * time.Hour).UTC(),
}

var JaxMock = &mocks.Jax{}
var DBMock = &mocks.DB{}

var testContext = &api.Context{
	Clients: &clients.Clients{Jax: JaxMock,
		DB: DBMock,
	},
}

var testServer = api.Server{
	Context: testContext,
}

func TestFilterAndSortFutureEvents(t *testing.T) {
	a := assert.New(t)

	Events := []*db.Event{
		&EventA,
		&EventB,
		&EventC,
		&EventD,
	}

	ctx := context.Background()
	DBMock.On("SelectEventExtensionsByEventIDs", ctx, mock.Anything).Return([]*db.EventExtension{}, nil)

	futureEvents, err := testServer.FilterAndSortFutureEvents(ctx, Events, 3)

	a.Nil(err)
	a.Equal(len(futureEvents), 3, "Incorrect amount of events returned")
	a.Equal(EventD.ID, futureEvents[0].ID, "EventD should be the first event")
	a.Equal(EventB.ID, futureEvents[1].ID, "EventB should be the second event")
	a.Equal(EventC.ID, futureEvents[2].ID, "EventC should be the third event")
}

func TestFilterAndSortPastEvents(t *testing.T) {
	a := assert.New(t)

	Events := []*db.Event{
		&EventA,
		&EventB,
		&EventC,
		&EventD,
	}

	ctx := context.Background()
	DBMock.On("SelectEventExtensionsByEventIDs", ctx, mock.Anything).Return([]*db.EventExtension{}, nil)

	pastEvents, err := testServer.FilterAndSortPastEvents(ctx, Events, 3)

	a.Nil(err)
	a.Equal(len(pastEvents), 3, "Incorrect amount of events returned")
	a.Equal(EventA.ID, pastEvents[0].ID, "EventA should be the first event")
	a.Equal(EventC.ID, pastEvents[1].ID, "EventC should be the second event")
	a.Equal(EventB.ID, pastEvents[2].ID, "EventB should be the third event")
}

func TestFilterAndSortLiveEvents(t *testing.T) {
	a := assert.New(t)

	Events := []*db.Event{
		&EventA,
		&EventB,
		&EventC,
		&EventD,
		&EventE,
	}

	highestConcurrent := 44444
	highConcurrent := 3333
	lowConcurrent := 22

	correctGameID := 1
	incorrectGameID := 2

	// EventA Live w/ 22 concurrents
	// EventB Live w/ 3333 concurrents
	// EventC Not Live
	// EventD Live w/ 44444 concurrents
	// EventE Live w/ 3333 concurrents on the wrong game
	jaxResponse := jax.Streams{
		TotalCount: 2,
		Hits: []*jax.Stream{
			{
				Channel: "blerg",
				Properties: jax.StreamProperties{
					Rails: &jax.StreamRailsProperties{
						ChannelID: &EventA.ChannelID,
						GameID:    &correctGameID,
					},
					Usher: &jax.StreamUsherProperties{
						ChannelCount: &lowConcurrent,
					},
				},
			},
			{
				Channel: "blargh",
				Properties: jax.StreamProperties{
					Rails: &jax.StreamRailsProperties{
						ChannelID: &EventB.ChannelID,
						GameID:    &correctGameID,
					},
					Usher: &jax.StreamUsherProperties{
						ChannelCount: &highConcurrent,
					},
				},
			},
			{
				Channel: "blinger",
				Properties: jax.StreamProperties{
					Rails: &jax.StreamRailsProperties{
						ChannelID: &EventD.ChannelID,
						GameID:    &correctGameID,
					},
					Usher: &jax.StreamUsherProperties{
						ChannelCount: &highestConcurrent,
					},
				},
			},
			{
				Channel: "blooba",
				Properties: jax.StreamProperties{
					Rails: &jax.StreamRailsProperties{
						ChannelID: &EventE.ChannelID,
						GameID:    &incorrectGameID,
					},
					Usher: &jax.StreamUsherProperties{
						ChannelCount: &highConcurrent,
					},
				},
			},
		},
	}

	ctx := context.Background()
	JaxMock.On("GetStreamsByChannelIDs", ctx, mock.Anything, mock.Anything, mock.Anything).Return(&jaxResponse, nil)
	DBMock.On("SelectEventExtensionsByEventID", ctx, mock.Anything).Return([]*db.EventExtension{}, nil)

	filteredEvents, err := testServer.FilterAndSortLiveEvents(ctx, Events, 3, correctGameID)
	a.Nil(err)

	a.Equal(2, len(filteredEvents), "Incorrect amount of events returned")
	a.Equal(EventD.ID, filteredEvents[0].ID, "EventD should be the first event")
	a.Equal(EventB.ID, filteredEvents[1].ID, "EventB should be the second event")
}
