package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"
	"time"

	log "github.com/Sirupsen/logrus"
	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/auth"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/internal/clients/tracking"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/common/goauthorization"
)

// V1UpdateEvent validates event attributes before updating the database record.
func (s *Server) V1UpdateEvent(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)
	now := time.Now()

	// Validate URL parameter "event_id":
	eventIDStr := pat.Param(r, "event_id")
	eventID, err := strconv.Atoi(eventIDStr)
	if err != nil || eventID < 0 {
		writer.BadRequest(fmt.Sprintf("Invalid event ID (%s)", eventIDStr))
		return
	}

	// Find existing event record by ID:
	selectedEvent, err := s.DB.SelectEventByID(r.Context(), eventID)
	if err != nil || selectedEvent == nil {
		writer.NotFound("Event record not found")
		return
	}

	// Validate JSON request body:
	var reqBody view.PutV1EventInput
	err = json.NewDecoder(r.Body).Decode(&reqBody)
	if err != nil {
		writer.BadRequest("Failed to decode JSON request body")
		return
	}

	err = reqBody.Validate()
	if err != nil {
		writer.BadRequest(err.Error())
		return
	}

	channelIDStr := strconv.Itoa(selectedEvent.ChannelID)

	// Make sure user has permission to edit the channel specified
	capabilities := goauthorization.CapabilityClaims{
		"manage_event": goauthorization.CapabilityClaim{
			"channel_id": channelIDStr,
		},
	}

	err = auth.AuthorizeToken(r, &capabilities)
	if err != nil {
		writer.Forbidden("User does not have permission to edit Event")
		return
	}

	// Validate parameter 'game_id'
	if selectedEvent.GameID != reqBody.GameID {
		_, err = s.Discovery.GetGameByID(r.Context(), reqBody.GameID)
		if err != nil {
			msg := fmt.Sprintf("Parameter 'game_id' (%d) is not a valid game ID.", reqBody.GameID)
			writer.BadRequest(msg)
			return
		}
	}

	// if request doesnt have an img id, but he had one before...
	//	we should hide the old image in s3
	if reqBody.CoverImageID == nil {
		if selectedEvent.CoverImageUUID != nil {
			err = s.S3.ExpireSavedImage(r.Context(), channelIDStr, *selectedEvent.CoverImageUUID)
			if err != nil {
				writer.Conflict("Failed to delete cover image.")
				return
			}
		}
	} else if selectedEvent.CoverImageUUID == nil || *reqBody.CoverImageID != *selectedEvent.CoverImageUUID {
		// Attempt to copy image from pending key prefix to saved key prefix, only if they're different
		err = s.S3.SaveFromPending(r.Context(), channelIDStr, *reqBody.CoverImageID)
		if err != nil {
			writer.BadRequest("Failed to save image.")
			return
		}

		idExists, err := s.DB.ExistingCoverImage(r.Context(), *reqBody.CoverImageID)
		if idExists {
			writer.BadRequest("Image UUID already exists.")
			return
		}

		if err != nil {
			writer.Conflict("Image UUID lookup failed.")
			return
		}
	}

	// Persist event attributes to database:
	eventUpdateAttributes := &db.Event{
		ID:             eventID,
		StartTimeUTC:   reqBody.StartTimeUTC.UTC(),
		EndTimeUTC:     reqBody.EndTimeUTC.UTC(),
		TimeZoneID:     reqBody.TimeZoneID,
		Title:          reqBody.Title,
		GameID:         reqBody.GameID,
		Description:    reqBody.Description,
		CoverImageUUID: reqBody.CoverImageID,
	}

	err = s.TMI.CensorEventInformation(r.Context(), eventUpdateAttributes)
	if err != nil {
		writer.Conflict("Failed to update event")
		return
	}

	updatedEvent, err := s.DB.UpdateEvent(r.Context(), eventUpdateAttributes)
	if err != nil {
		writer.Conflict("Failed to update event record")
		return
	} else if updatedEvent == nil {
		writer.Gone("Event doesn't exist anymore")
		return
	}

	// Expire caches
	go func() {
		contextLogger := log.WithFields(log.Fields{
			"method":   r.Method,
			"url":      r.RequestURI,
			"event_id": eventID,
		})

		err = s.Cache.ExpireEventView(eventID)
		if err != nil {
			contextLogger.WithError(err).Warn("api: failed to expire cached event view")
		}

		err = s.Cache.ExpireListEventsView(selectedEvent.ChannelID)
		if err != nil {
			contextLogger.WithError(err).Warn("api: failed to expire cached list events view")
		}
	}()

	// Send tracking data:
	var userPtr *int
	token, err := auth.GetToken(r)
	if err == nil && token != nil {
		userID, err := strconv.Atoi(token.GetSubject())
		if err == nil {
			userPtr = &userID
		}
	}

	var updatedAtTimestampPtr *float64
	if updatedEvent.UpdatedAtUTC != nil {
		updatedAtTimestamp := float64(updatedEvent.UpdatedAtUTC.Unix())
		updatedAtTimestampPtr = &updatedAtTimestamp
	}

	trackingData := tracking.OracleServerEvent{
		Action:                 tracking.OracleServerUpdateAction,
		ServerTimestamp:        float64(now.Unix()),
		UserID:                 userPtr,
		EventID:                updatedEvent.ID,
		ChannelID:              updatedEvent.ChannelID,
		EventStartTimestamp:    float64(updatedEvent.StartTimeUTC.Unix()),
		EventEndTimestamp:      float64(updatedEvent.EndTimeUTC.Unix()),
		Timezone:               updatedEvent.TimeZoneID,
		Title:                  updatedEvent.Title,
		GameID:                 updatedEvent.GameID,
		Description:            updatedEvent.Description,
		Status:                 updatedEvent.Status,
		CreatedAtTimestamp:     float64(updatedEvent.CreatedAtUTC.Unix()),
		UpdatedAtTimestamp:     updatedAtTimestampPtr,
		CoverImageUUID:         updatedEvent.CoverImageUUID,
		FallbackCoverImageUUID: updatedEvent.FallbackCoverImageUUID,
	}

	go s.Tracking.SendServerActionEvent(r.Context(), &trackingData)

	// Format the response payload:
	payload := &view.PutV1EventOutput{
		Status:  http.StatusOK,
		Message: "Successfully updated event record.",
		Meta: view.PutV1EventOutputMeta{
			Status: updatedEvent.Status,
		},
		Data: view.V1EventView{
			ID:                    updatedEvent.ID,
			ChannelID:             updatedEvent.ChannelID,
			StartTimeUTC:          updatedEvent.StartTimeUTC,
			EndTimeUTC:            updatedEvent.EndTimeUTC,
			TimeZoneID:            updatedEvent.TimeZoneID,
			Title:                 updatedEvent.Title,
			Description:           updatedEvent.Description,
			GameID:                updatedEvent.GameID,
			CreatedAtUTC:          updatedEvent.CreatedAtUTC,
			UpdatedAtUTC:          updatedEvent.UpdatedAtUTC,
			CoverImageID:          updatedEvent.CoverImageUUID,
			CoverImageSourceURL:   updatedEvent.CoverImageSourceURL(),
			CoverImageURLTemplate: updatedEvent.CoverImageURLTemplate(),
		},
	}

	writer.OK(payload)
}
