package api

import (
	"encoding/json"
	"net/http"
	"strconv"

	log "github.com/Sirupsen/logrus"
	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/view"
)

// V1UpdateEventExtension creates a new event for a user given the PutV1EventExtensionInput struct
func (s *Server) V1UpdateEventExtension(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)

	eventIDStr := pat.Param(r, "event_id")
	eventID, err := strconv.Atoi(eventIDStr)
	if err != nil {
		writer.BadRequest("Failed to decode event_id")
		return
	}

	key := pat.Param(r, "key")

	// Validate event.
	_, err = s.DB.SelectEventByID(r.Context(), eventID)
	if err != nil {
		writer.NotFound("Event extension not found")
		return
	}

	// Get event extension.
	eventExtension, err := s.DB.SelectEventExtensionByEventIDAndKey(r.Context(), eventID, key)
	if err != nil {
		writer.NotFound("Event extension not found")
		return
	}

	// Validate JSON request body:
	var reqBody view.PutV1EventExtensionInput
	err = json.NewDecoder(r.Body).Decode(&reqBody)
	if err != nil {
		writer.BadRequest("Failed to decode JSON request body")
		return
	}

	err = reqBody.Validate()
	if err != nil {
		writer.BadRequest(err.Error())
		return
	}

	eventExtension.Value = reqBody.Value

	savedEventExtension, err := s.DB.UpdateEventExtension(r.Context(), eventExtension)
	if err != nil || savedEventExtension == nil {
		writer.Conflict("Failed to save event extension record")
		return
	}

	// Expire cache
	go func() {
		err = s.Cache.ExpireListEventExtensionsView(eventID)
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"method":   r.Method,
				"url":      r.RequestURI,
				"event_id": savedEventExtension.EventID,
			}).Warn("api: failed to expire cached list event extensions view")
		}
	}()

	// Format JSON response payload:
	payload := &view.PutV1EventExtensionOutput{
		Status:  http.StatusOK,
		Message: "Successfully saved event extension record.",
	}

	writer.OK(payload)
}
