package api

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"
	"time"

	log "github.com/Sirupsen/logrus"
	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/auth"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/internal/clients/dynamodb"
	"code.justin.tv/cb/oracle/internal/clients/tracking"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/common/goauthorization"
)

// V1UpdateUserEventNotificationSettings updates event notification settings
// for a user.
//
// If the notification is set to TRUE, a record is updated or insert in
// the data store.
//
// If the notification is set to FALSE, an existing record is deleted,
// unsubscribing the user from the event's notification(s).
func (s *Server) V1UpdateUserEventNotificationSettings(w http.ResponseWriter, r *http.Request) {
	writer := responder.NewResponseWriter(w)
	now := time.Now()

	// Validate PUT request parameters:
	eventIDStr := pat.Param(r, "event_id")
	eventID, err := strconv.Atoi(eventIDStr)
	if err != nil || eventID <= 0 {
		writer.BadRequest(fmt.Sprintf("Invalid event ID (%s)", eventIDStr))
		return
	}

	userIDStr := pat.Param(r, "user_id")
	userID, err := strconv.Atoi(userIDStr)
	if err != nil || userID <= 0 {
		writer.BadRequest(fmt.Sprintf("Invalid user ID (%s)", userIDStr))
		return
	}

	// Validate PUT request body:
	var reqBody view.PutV1UserEventNotificationSettingsInput
	err = json.NewDecoder(r.Body).Decode(&reqBody)
	if err != nil {
		writer.BadRequest("Failed to decode JSON request body")
		return
	}

	// Find existing event record by ID:
	selectedEvent, err := s.DB.SelectEventByID(r.Context(), eventID)
	if err != nil || selectedEvent == nil {
		msg := fmt.Sprintf("Event record (ID %s) not found", eventIDStr)
		writer.NotFound(msg)
		return
	} else if selectedEvent.Status != db.EventStatusAvailable {
		msg := fmt.Sprintf("Event (ID %s) not available for notification", eventIDStr)
		writer.BadRequest(msg)
		return
	}

	// Validate the user's permission to update the user's event notification:
	requestingUserIDStr := r.URL.Query().Get("user_id")
	if requestingUserIDStr != "" {
		if requestingUserIDStr != userIDStr {
			writer.Forbidden("User does not have permission to edit notifications")
			return
		}
	} else {
		// Fall back to cartman capabilities
		capabilities := goauthorization.CapabilityClaims{
			"manage_user_event_notification_settings": goauthorization.CapabilityClaim{
				"user_id": userIDStr,
			},
		}

		err = auth.AuthorizeToken(r, &capabilities)
		if err != nil {
			writer.Forbidden("User does not have permission to edit notifications")
			return
		}
	}

	// Disallow update if event is in the past:
	if selectedEvent.EndTimeUTC.Before(time.Now()) {
		writer.UnprocessableEntity("Cannot update notification settings for a past event")
		return
	}

	// Update user's email notification setting:
	if reqBody.EmailEnabled {
		err = s.subscribeToEmailNotification(r.Context(), eventID, userID)
	} else {
		err = s.unsubscribeToEmailNotification(r.Context(), eventID, userID)
	}
	if err != nil {
		msg := fmt.Sprintf("Failed to update notification (event ID %d, user ID %d)", eventID, userID)
		writer.InternalServerError(msg, err)
		return
	}

	// Send tracking data:
	go func() {
		var currentUserIDPtr *int

		if requestingUserIDStr != "" {
			currentUserID, err := strconv.Atoi(requestingUserIDStr)
			if err == nil {
				currentUserIDPtr = &currentUserID
			}
		}

		if currentUserIDPtr == nil {
			token, err := auth.GetToken(r)
			if err == nil && token != nil {
				currentUserID, err := strconv.Atoi(token.GetSubject())
				if err == nil {
					currentUserIDPtr = &currentUserID
				}
			}
		}

		trackingData := tracking.OracleNotificationEvent{
			ServerTimestamp:   float64(now.Unix()),
			UserID:            currentUserIDPtr,
			EventID:           selectedEvent.ID,
			EmailNotification: reqBody.EmailEnabled,
		}

		s.Tracking.SendNotificationChangeEvent(r.Context(), &trackingData)
	}()

	// Format PUT response payload:
	payload := &view.PutV1UserEventNotificationSettingsOutput{
		Status:  http.StatusOK,
		Message: fmt.Sprintf("Successfully updated notification (event ID %d, user ID %d)", eventID, userID),
		Data: view.PutV1UserEventNotificationSettingsOutputData{
			EmailEnabled: reqBody.EmailEnabled,
		},
	}

	writer.OK(payload)
}

func (s *Server) subscribeToEmailNotification(ctx context.Context, eventID int, userID int) error {
	userIDStr := strconv.Itoa(userID)

	// Query from DynamoDB:
	notif, err := s.DynamoDB.GetEventNotificationsForUser(ctx, eventID, userIDStr)
	if err != nil && err != dynamodb.ErrNoQueryItems {
		return err
	}

	// If record exists then no-op
	if notif != nil {
		return nil
	}

	// Update user's notification setting:
	input := dynamodb.SubscriptionItem{
		EventID: eventID,
		UserID:  userIDStr,
		Email:   true,
	}

	err = s.DynamoDB.SubscribeEvent(ctx, input)
	if err != nil {
		return err
	}

	_, err = s.DB.IncrementEventEmailNotificationCount(ctx, eventID)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"event_id":      eventID,
			"user_id":       userID,
			"email_enabled": true,
		}).Error("api: failed to increment event email notification count")
	}

	return nil
}

func (s *Server) unsubscribeToEmailNotification(ctx context.Context, eventID int, userID int) error {
	userIDStr := strconv.Itoa(userID)

	// Query from DynamoDB:
	_, err := s.DynamoDB.GetEventNotificationsForUser(ctx, eventID, userIDStr)
	switch {
	case err == dynamodb.ErrNoQueryItems:
		// If record does NOT exist then no-op
		return nil
	case err != nil:
		return err
	}

	// Update user's notification setting:
	input := dynamodb.SubscriptionItem{
		UserID:  userIDStr,
		EventID: eventID,
		Email:   false,
	}

	err = s.DynamoDB.UnsubscribeEvent(ctx, input)
	if err != nil {
		return err
	}

	_, err = s.DB.DecrementEventEmailNotificationCount(ctx, eventID)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"event_id":      eventID,
			"user_id":       userID,
			"email_enabled": false,
		}).Error("api: failed to decrement event email notification count")
	}

	return nil
}
