package api

import (
	"encoding/json"
	"net/http"

	"goji.io/pat"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/auth"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/common/goauthorization"
)

// V1UploadChannelEventCoverImage validates the base64-encoded image file in
// the request body before uploading the file to a pending S3 storage.
func (s *Server) V1UploadChannelEventCoverImage(w http.ResponseWriter, r *http.Request) {
	s.uploadChannelEventCoverImageV1(w, r, true)
}

func (s *Server) uploadChannelEventCoverImageV1(w http.ResponseWriter, r *http.Request, validateToken bool) {
	writer := responder.NewResponseWriter(w)

	channelIDStr := pat.Param(r, "channel_id")

	// Validate JSON request body:
	var reqBody view.PostV1ChannelEventCoverImageInput
	err := json.NewDecoder(r.Body).Decode(&reqBody)
	if err != nil {
		writer.BadRequest("Failed to decode JSON request body")
		return
	}

	if reqBody.Base64EncodedImage == "" {
		writer.BadRequest("Parameter 'event_cover_image' is required.")
		return
	}

	if validateToken {
		// Validate authorization token:
		capabilities := goauthorization.CapabilityClaims{
			"manage_event": goauthorization.CapabilityClaim{
				"channel_id": channelIDStr,
			},
		}

		err = auth.AuthorizeToken(r, &capabilities)
		if err != nil {
			writer.Forbidden("Not authorized to upload event cover image")
			return
		}
	}

	// Validate image requirements and upload to S3:
	versionID, err := s.S3.ValidateAndUploadCoverImage(r.Context(), channelIDStr, reqBody.Base64EncodedImage)
	if err != nil {
		writer.UnprocessableEntity(err.Error())
		return
	}

	// Format JSON response body:
	payload := &view.PostV1ChannelEventCoverImageOutput{
		Status:  http.StatusOK,
		Message: "Successfully uploaded event cover image",
		Data: view.PostV1ChannelEventCoverImageOutputData{
			ID: *versionID,
		},
	}

	writer.OK(payload)
}
