package auth

import (
	"io/ioutil"
	"net/http"

	"code.justin.tv/cb/oracle/config"
	"code.justin.tv/common/goauthorization"
)

// we will store a local instance of decoder that we will use ot authorize tokenss
var privateDecoder goauthorization.Decoder

const (
	algorithm = "ES256"
	audience  = "code.justin.tv/cb/oracle"
	issuer    = "code.justin.tv/web/cartman"
	keyPath   = "/var/app/ecc_public.key"
)

// InitializeDecoder creates a new decoder object and stores it locally
func InitializeDecoder() error {
	if config.Environment == "development" {
		return nil
	}

	key, err := ioutil.ReadFile(keyPath)
	if err != nil {
		return err
	}

	d, err := goauthorization.NewDecoder(algorithm, audience, issuer, key)
	if err != nil {
		return err
	}

	privateDecoder = d
	return nil
}

//	get the raw authorizationtoken object (this maybe important for retrieving
//	the subject for instance)
func GetToken(r *http.Request) (*goauthorization.AuthorizationToken, error) {
	if config.Environment == "development" {
		return nil, nil
	}

	token, err := privateDecoder.ParseToken(r)
	if err != nil {
		return nil, err
	}

	return token, nil
}

// AuthorizeToken, given a validation token embedded in the req header, check that
// the request is authorized for a given capability.
func AuthorizeToken(r *http.Request, c *goauthorization.CapabilityClaims) error {
	if config.Environment == "development" {
		return nil
	}

	token, err := privateDecoder.ParseToken(r)
	if err != nil {
		return err
	}

	return privateDecoder.Validate(token, *c)
}
