package cache

import (
	"encoding/json"
	"fmt"
	"time"

	"code.justin.tv/cb/oracle/config"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/view"
	log "github.com/Sirupsen/logrus"
	"github.com/bradfitz/gomemcache/memcache"
)

const (
	EventViewExpiration      = 60      // 1 minute
	ListEventsViewExpiration = 60      // 1 minute
	NotificationExpiration   = 60 * 10 // 10 minutes

	Timeout            = 2 * time.Second
	MaxIdleConnections = 30

	CacheTimeFormat = "2006-01-02_15_04_00"
)

type CacheStruct struct {
	store *memcache.Client
}

func NewAPICache() *CacheStruct {
	client := memcache.New(config.Values.ApiCacheHostPort)
	client.Timeout = Timeout
	client.MaxIdleConns = MaxIdleConnections

	return &CacheStruct{
		store: client,
	}
}

func NewCronCache() *CacheStruct {
	return &CacheStruct{
		store: memcache.New(config.Values.CronCacheHostPort),
	}
}

func (c *CacheStruct) GetUserNotificationDelivery(interval string, eventID int, userID string) (*[]byte, error) {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.userNotificationDeliveryCacheKey(interval, eventID, userID)
	cacheItem, err := c.store.Get(key)
	if err == memcache.ErrCacheMiss {
		return nil, nil
	} else if err != nil {
		log.WithError(err).WithField("key", key).Error("cache: failed to get user notification delivery view")
		return nil, err
	}
	return &cacheItem.Value, nil
}

func (c *CacheStruct) CacheUserNotificationDelivery(interval string, eventID int, userID string) error {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.userNotificationDeliveryCacheKey(interval, eventID, userID)
	bytes, err := json.Marshal(eventID)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":  key,
			"view": eventID,
		}).Error("cache: failed to JSON marshal user notification delivery view")

		return err
	}

	err = c.store.Set(&memcache.Item{
		Key:        key,
		Value:      bytes,
		Expiration: NotificationExpiration,
	})

	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":        key,
			"value":      string(bytes),
			"expiration": NotificationExpiration,
		}).Error("cache: failed to set user notification delivery view")

		return err
	}

	return nil
}

func (c *CacheStruct) userNotificationDeliveryCacheKey(interval string, eventID int, userID string) string {
	return fmt.Sprintf("eventnotification/%s/%d/%s", interval, eventID, userID)
}

func (c *CacheStruct) GetNotificationDelivery(interval string, eventID int) (*[]byte, error) {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.notificationDeliveryCacheKey(interval, eventID)

	cacheItem, err := c.store.Get(key)
	if err == memcache.ErrCacheMiss {
		return nil, nil
	} else if err != nil {
		log.WithError(err).WithField("key", key).Error("cache: failed to get notification delivery view")
		return nil, err
	}
	return &cacheItem.Value, nil
}

func (c *CacheStruct) CacheNotificationDelivery(interval string, eventID int) error {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.notificationDeliveryCacheKey(interval, eventID)

	bytes, err := json.Marshal(eventID)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":  key,
			"view": eventID,
		}).Error("cache: failed to JSON marshal notification delivery view")

		return err
	}

	err = c.store.Set(&memcache.Item{
		Key:        key,
		Value:      bytes,
		Expiration: NotificationExpiration,
	})

	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":        key,
			"value":      string(bytes),
			"expiration": NotificationExpiration,
		}).Error("cache: failed to set notification delivery view")

		return err
	}

	return nil
}

func (c *CacheStruct) notificationDeliveryCacheKey(interval string, eventID int) string {
	return fmt.Sprintf("eventnotification/%s/%d", interval, eventID)
}

func (c *CacheStruct) GetEventView(eventID int) (*[]byte, error) {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.eventViewCacheKey(eventID)
	cacheItem, err := c.store.Get(key)
	if err == memcache.ErrCacheMiss {
		return nil, nil
	} else if err != nil {
		log.WithError(err).WithField("key", key).Error("cache: failed to get event view")
		return nil, err
	}

	return &cacheItem.Value, nil
}

func (c *CacheStruct) CacheEventView(eventID int, eventView view.GetV1EventOutput) error {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.eventViewCacheKey(eventID)
	bytes, err := json.Marshal(eventView)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":  key,
			"view": eventView,
		}).Error("cache: failed to JSON marshal event view")

		return err
	}

	err = c.store.Set(&memcache.Item{
		Key:        key,
		Value:      bytes,
		Expiration: EventViewExpiration,
	})

	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":        key,
			"value":      string(bytes),
			"expiration": EventViewExpiration,
		}).Error("cache: failed to set event view")

		return err
	}

	return nil
}

func (c *CacheStruct) ExpireEventView(eventID int) error {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.eventViewCacheKey(eventID)
	err := c.store.Delete(key)
	if err != nil && err != memcache.ErrCacheMiss {
		log.WithError(err).WithField("key", key).Error("cache: failed to delete event view")
		return err
	}

	return nil
}

func (c *CacheStruct) eventViewCacheKey(eventID int) string {
	return fmt.Sprintf("eventview/%d", eventID)
}

// Get List Events view response that expires within 60 seconds.
func (c *CacheStruct) GetListEventsView(params db.EventListParams) (*[]byte, error) {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.listEventsViewCacheKey(params)
	cacheItem, err := c.store.Get(key)
	if err == memcache.ErrCacheMiss {
		return nil, nil
	} else if err != nil {
		log.WithError(err).WithField("key", key).Error("cache: failed to get list events view")
		return nil, err
	}

	return &cacheItem.Value, nil
}

// CacheListEventsView caches list events view for 60 seconds.
func (c *CacheStruct) CacheListEventsView(params db.EventListParams, listEventsView view.GetV1AvailableEventListOutput) error {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.listEventsViewCacheKey(params)
	bytes, err := json.Marshal(listEventsView)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":  key,
			"view": listEventsView,
		}).Error("cache: failed to JSON marshal list events view")

		return err
	}

	err = c.store.Set(&memcache.Item{
		Key:        key,
		Value:      bytes,
		Expiration: ListEventsViewExpiration,
	})

	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":        key,
			"value":      string(bytes),
			"expiration": ListEventsViewExpiration,
		}).Error("cache: failed to set list events view")

		return err
	}

	return nil
}

func (c *CacheStruct) ExpireListEventsView(channelID int) error {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	timeNow := time.Now()
	timeZero := time.Unix(0, 0)
	timeInf := timeNow.AddDate(1000, 0, 0) // one thousand years from now

	// try your best to expire all possible caches
	for i := 0; i < 5; i++ {
		// Future view
		paramsFuture := db.EventListParams{
			ChannelID:     channelID,
			EndTimeBefore: timeInf.Add(time.Duration(-i) * time.Minute),
			EndTimeAfter:  timeNow.Add(time.Duration(-i) * time.Minute),
			OrderBy:       "asc",
		}

		futureKey := c.listEventsViewCacheKey(paramsFuture)
		err := c.store.Delete(futureKey)
		if err != nil && err != memcache.ErrCacheMiss {
			log.WithError(err).WithField("key", futureKey).Error("cache: failed to delete list events view")
			return err
		}

		// Past view
		paramsPast := db.EventListParams{
			ChannelID:     channelID,
			EndTimeBefore: timeNow.Add(time.Duration(-i) * time.Minute),
			EndTimeAfter:  timeZero,
			OrderBy:       "desc",
		}

		pastKey := c.listEventsViewCacheKey(paramsPast)
		err = c.store.Delete(pastKey)
		if err != nil && err != memcache.ErrCacheMiss {
			log.WithError(err).WithField("key", pastKey).Error("cache: failed to delete list events view")
			return err
		}
	}

	return nil
}

func (c *CacheStruct) listEventsViewCacheKey(params db.EventListParams) string {
	return fmt.Sprintf(
		"list_events_view/%d/%s/%s/%s",
		params.ChannelID,
		params.EndTimeAfter.Format(CacheTimeFormat),
		params.EndTimeBefore.Format(CacheTimeFormat),
		params.OrderBy,
	)
}
