package cache

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/cb/oracle/view"
	log "github.com/Sirupsen/logrus"
	"github.com/bradfitz/gomemcache/memcache"
)

// Get List Event Extensions view response that expires within 60 seconds.
func (c *CacheStruct) GetListEventExtensionsView(eventID int) (*[]byte, error) {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.listEventExtensionsViewCacheKey(eventID)
	cacheItem, err := c.store.Get(key)
	if err == memcache.ErrCacheMiss {
		return nil, nil
	} else if err != nil {
		log.WithError(err).WithField("key", key).Error("cache: failed to get list event extensions view")
		return nil, err
	}

	return &cacheItem.Value, nil
}

// Cache list Event Extensions view for 60 seconds.
func (c *CacheStruct) CacheListEventExtensionsView(eventID int, listEventExtensionsView view.GetV1EventExtensionListOutput) error {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.listEventExtensionsViewCacheKey(eventID)
	bytes, err := json.Marshal(listEventExtensionsView)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":  key,
			"view": listEventExtensionsView,
		}).Error("cache: failed to JSON marshal event extension view")

		return err
	}

	err = c.store.Set(&memcache.Item{
		Key:        key,
		Value:      bytes,
		Expiration: EventViewExpiration,
	})

	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":        key,
			"value":      string(bytes),
			"expiration": EventViewExpiration,
		}).Error("cache: failed to set event view")

		return err
	}

	return nil
}

// Expire list Event Extensions for channel
func (c *CacheStruct) ExpireListEventExtensionsView(eventID int) error {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.listEventExtensionsViewCacheKey(eventID)
	err := c.store.Delete(key)
	if err != nil && err != memcache.ErrCacheMiss {
		log.WithError(err).WithField("key", key).Error("cache: failed to delete event extensions view")
		return err
	}

	return nil
}

func (c *CacheStruct) listEventExtensionsViewCacheKey(eventID int) string {
	return fmt.Sprintf(
		"list_events_extensions_view/%d",
		eventID,
	)
}
