package cache

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/cb/oracle/view"
	log "github.com/Sirupsen/logrus"
	"github.com/bradfitz/gomemcache/memcache"
)

const (
	GameEventsViewExpiration = 60 * 1
)

// Get Game Events view response that expires within 60 seconds.
func (c *CacheStruct) GetGameEventsView(gameID int) (*[]byte, error) {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.gameEventsKey(gameID)
	cacheItem, err := c.store.Get(key)
	if err == memcache.ErrCacheMiss {
		return nil, nil
	} else if err != nil {
		log.WithError(err).WithField("key", key).Error("cache: failed to get list events view")
		return nil, err
	}

	return &cacheItem.Value, nil
}

// CacheGameEventsView caches list events view for 60 seconds.
func (c *CacheStruct) CacheGameEventsView(gameID int, eventSuggestions view.GetV1EventSuggestionsOutput) error {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.gameEventsKey(gameID)
	bytes, err := json.Marshal(eventSuggestions)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":  key,
			"view": eventSuggestions,
		}).Error("cache: failed to JSON marshal list events view")

		return err
	}

	err = c.store.Set(&memcache.Item{
		Key:        key,
		Value:      bytes,
		Expiration: GameEventsViewExpiration,
	})

	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key":        key,
			"value":      string(bytes),
			"expiration": GameEventsViewExpiration,
		}).Error("cache: failed to set list events view")

		return err
	}

	return nil
}

// ExpireGameEventsView expire views by game.
func (c *CacheStruct) ExpireGameEventsView(gameID int) error {
	defer func() { //catch or finally
		if err := recover(); err != nil { //catch
			log.Error(err)
		}
	}()

	key := c.gameEventsKey(gameID)
	err := c.store.Delete(key)

	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"key": key,
		}).Error("cache: failed to expire game events view")

		return err
	}

	return nil
}

func (c *CacheStruct) gameEventsKey(gameID int) string {
	return fmt.Sprintf("game_events/%d", gameID)
}
