package clients

import (
	log "github.com/Sirupsen/logrus"

	"code.justin.tv/foundation/twitchclient"
	jax "code.justin.tv/web/jax/client"

	"code.justin.tv/cb/oracle/config"
	"code.justin.tv/cb/oracle/internal/clients/cache"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/internal/clients/db/pq"
	"code.justin.tv/cb/oracle/internal/clients/discovery"
	"code.justin.tv/cb/oracle/internal/clients/dynamodb"
	"code.justin.tv/cb/oracle/internal/clients/s3"
	"code.justin.tv/cb/oracle/internal/clients/sns"
	"code.justin.tv/cb/oracle/internal/clients/stats"
	"code.justin.tv/cb/oracle/internal/clients/tmi"
	"code.justin.tv/cb/oracle/internal/clients/tracking"
	"code.justin.tv/cb/oracle/internal/clients/users"
	"github.com/cactus/go-statsd-client/statsd"
)

// Clients contains all client interfaces to external resources
// required by the application.
type Clients struct {
	Cache     cache.APICache
	CronCache cache.CronCache
	DB        db.DB
	Discovery discovery.Service
	DynamoDB  dynamodb.Client
	Jax       jax.Client
	S3        s3.S3i
	SNS       sns.SNSi
	Stats     statsd.Statter
	TMI       tmi.Service
	Tracking  tracking.Service
	Users     users.Service
}

// NewClients instantiates all necessary clients and returns
// a wrapper Clients struct.
func NewClients() (*Clients, error) {
	log.Info("Instantiating new client interfaces")

	db, err := pq.OpenConnection(config.Secrets.DBCredentials, config.Values.DBAddress)
	if err != nil {
		return nil, err
	}

	dynamoClient, err := dynamodb.NewDynamoClient(config.Environment)
	if err != nil {
		return nil, err
	}

	jaxClient, err := jax.NewClient(twitchclient.ClientConf{
		Host: config.Values.JaxServiceHost,
	})
	if err != nil {
		return nil, err
	}

	s3Client, err := s3.NewS3Client(config.Environment)
	if err != nil {
		return nil, err
	}

	snsClient, err := sns.NewSNSClient(config.Environment)
	if err != nil {
		return nil, err
	}

	st, err := stats.NewStatsClient("oracle-api")
	if err != nil {
		return nil, err
	}

	tmiClient, err := tmi.New()
	if err != nil {
		return nil, err
	}

	trackingClient, err := tracking.NewClient()
	if err != nil {
		return nil, err
	}

	clientsWrapper := &Clients{
		Cache:     cache.NewAPICache(),
		CronCache: cache.NewCronCache(),
		DB:        db,
		Discovery: discovery.NewHTTPClient(config.Values.DiscoveryServiceHost),
		DynamoDB:  dynamoClient,
		Jax:       jaxClient,
		S3:        s3Client,
		SNS:       snsClient,
		Stats:     st,
		TMI:       tmiClient,
		Tracking:  trackingClient,
		Users:     users.NewHTTPClient(config.Values.UsersServiceHost),
	}

	return clientsWrapper, nil
}
