package db

import (
	"fmt"
	"math/rand"
	"strconv"
	"time"

	"code.justin.tv/cb/oracle/config"
	"code.justin.tv/cb/oracle/internal/clients/s3"
)

const SelectLimit = 100

// The strings below are the allowed statuses for an `event` record.
const (
	EventStatusAvailable      = "available"
	EventStatusCreatorDeleted = "creator_deleted"
	EventStatusAdminDeleted   = "admin_deleted"
)

// Event contains the column fields from the database's `events` table
type Event struct {
	ID                     int
	ChannelID              int
	StartTimeUTC           time.Time
	EndTimeUTC             time.Time
	TimeZoneID             string
	Title                  string
	GameID                 int
	Description            *string
	Status                 string
	CreatedAtUTC           time.Time
	UpdatedAtUTC           *time.Time
	CoverImageUUID         *string
	FallbackCoverImageUUID *string

	// Optional fields only populated if event was queried by extension
	ExtensionField *string
	ExtensionValue *string
}

// ManagerEvent contains the column fields from the database,
// joining the `events` table and the `event_stats` table.
type ManagerEvent struct {
	// `events` table
	Event

	// `event_stats` table
	EnabledUserEmailNotificationCount int
}

// EventListParams is used as an input for filtering the database query
// via WHERE and ORDER BY clauses.
type EventListParams struct {
	ChannelID     int
	EndTimeBefore time.Time
	EndTimeAfter  time.Time
	OrderBy       string
}

// EventSelectionParams is used as an input for paginating the database query
type EventSelectionParams struct {
	Limit  *int
	Cursor *string
}

// EventSelectionParams is used as an output for including pagination info
type EventSelectionResponse struct {
	Events []*Event
	Cursor string
}

const (
	coverImageSourceURLFormat   = "https://%s.s3.amazonaws.com/%s/%s/%s"
	coverImageURLTemplateFormat = "https://static-cdn.jtvnw.net/%s/%s/%s-{width}x{height}"

	twitchEventURLFormat = "https://www.twitch.tv/events/%d"

	presetCoverImagePartition = "defaults"
)

// presetCoverImageIDs contains hard-coded cover image IDs that map to
// image assets stored in S3.
//
// These assets are used to randomly assign to Twitch Events that do not
// have custom cover images.
//
// Preset cover images are PERMANENTLY saved under the partition
// presetCoverImagePartition ("defaults").
var presetCoverImageIDs = map[string]struct{}{
	"city":     struct{}{},
	"farm":     struct{}{},
	"hills":    struct{}{},
	"pyramids": struct{}{},
	"town":     struct{}{},
}

// RandomPresetCoverImageID selects a cover image ID from presetCoverImageIDs
// at random.
func RandomPresetCoverImageID() string {
	rand.Seed(time.Now().UTC().UnixNano())

	count := len(presetCoverImageIDs)
	ids := make([]string, 0, count)

	for id := range presetCoverImageIDs {
		ids = append(ids, id)
	}

	return ids[rand.Intn(count)]
}

// CoverImageSourceURL returns the source URL of the cover image object
// in storage.
//
// CoverImageUUID is prioritized over FallbackCoverImageUUID.
//
// If CoverImageUUID is not present, and FallbackCoverImageUUID is either
// not present (from database) or not a valid preset cover image ID,
// then the returned source URL is nil.
func (e *Event) CoverImageSourceURL() *string {
	bucket := config.Values.S3CoverImageBucket
	prefix := s3.PrefixImagesSaved
	var partition string
	var id string

	if e.CoverImageUUID != nil {
		if e.Status != EventStatusAvailable {
			prefix = s3.PrefixImagesExpiring
		}

		partition = strconv.Itoa(e.ChannelID)
		id = *e.CoverImageUUID
	} else {
		if e.FallbackCoverImageUUID == nil {
			return nil
		}

		if _, ok := presetCoverImageIDs[*e.FallbackCoverImageUUID]; !ok {
			return nil
		}

		partition = presetCoverImagePartition
		id = *e.FallbackCoverImageUUID
	}

	url := fmt.Sprintf(coverImageSourceURLFormat, bucket, prefix, partition, id)

	return &url
}

// CoverImageURLTemplate returns the URL template to the cover image from CDN.
//
// CoverImageUUID is prioritized over FallbackCoverImageUUID.
//
// If CoverImageUUID is not present, and FallbackCoverImageUUID is either
// not present (from database) or not a valid preset cover image ID,
// then the returned URL template is nil.
func (e *Event) CoverImageURLTemplate() *string {
	prefix := config.Values.CDNCoverImagePrefix
	var partition string
	var id string

	if e.CoverImageUUID != nil {
		partition = strconv.Itoa(e.ChannelID)
		id = *e.CoverImageUUID
	} else {
		if e.FallbackCoverImageUUID == nil {
			return nil
		}

		if _, ok := presetCoverImageIDs[*e.FallbackCoverImageUUID]; !ok {
			return nil
		}

		partition = presetCoverImagePartition
		id = *e.FallbackCoverImageUUID
	}

	url := fmt.Sprintf(coverImageURLTemplateFormat, prefix, partition, id)

	return &url
}

// URL returns the production URL of a Twitch Event.
func (e *Event) URL() string {
	return fmt.Sprintf(twitchEventURLFormat, e.ID)
}
