package db

import (
	"testing"

	"code.justin.tv/cb/oracle/config"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
)

func TestRandomPresetCoverImageID(t *testing.T) {
	assert.Contains(t, presetCoverImageIDs, RandomPresetCoverImageID())
}

type CoverImageSourceURLSuite struct {
	suite.Suite
}

func TestCoverImageSourceURLSuite(t *testing.T) {
	suite.Run(t, &CoverImageSourceURLSuite{})
}

func (suite *CoverImageSourceURLSuite) SetupTest() {
	config.Values.S3CoverImageBucket = "some-bucket"
}

func (suite *CoverImageSourceURLSuite) TestCoverImageSourceURL_Nil() {
	event := Event{
		CoverImageUUID:         nil,
		FallbackCoverImageUUID: nil,
	}

	assert.Nil(suite.T(), event.CoverImageSourceURL())
}

func (suite *CoverImageSourceURLSuite) TestCoverImageSourceURL_CustomSaved() {
	coverImageUUID := "randomly-generated-uuid"
	event := &Event{
		Status:                 EventStatusAvailable,
		ChannelID:              123,
		CoverImageUUID:         &coverImageUUID,
		FallbackCoverImageUUID: nil,
	}

	url := event.CoverImageSourceURL()

	if assert.NotNil(suite.T(), url) {
		assert.Equal(suite.T(), "https://some-bucket.s3.amazonaws.com/images-saved/123/randomly-generated-uuid", *url)
	}
}

func (suite *CoverImageSourceURLSuite) TestCoverImageSourceURL_CustomExpiring() {
	coverImageUUID := "randomly-generated-uuid"
	event := &Event{
		Status:                 EventStatusAdminDeleted,
		ChannelID:              321,
		CoverImageUUID:         &coverImageUUID,
		FallbackCoverImageUUID: nil,
	}

	url := event.CoverImageSourceURL()

	if assert.NotNil(suite.T(), url) {
		assert.Equal(suite.T(), "https://some-bucket.s3.amazonaws.com/images-expiring/321/randomly-generated-uuid", *url)
	}
}

func (suite *CoverImageSourceURLSuite) TestCoverImageSourceURL_Fallback() {
	fallbackCoverImageUUID := "city"
	event := Event{
		CoverImageUUID:         nil,
		FallbackCoverImageUUID: &fallbackCoverImageUUID,
	}

	url := event.CoverImageSourceURL()

	if assert.NotNil(suite.T(), url) {
		assert.Equal(suite.T(), "https://some-bucket.s3.amazonaws.com/images-saved/defaults/city", *url)
	}
}

func (suite *CoverImageSourceURLSuite) TestCoverImageSourceURL_InvalidFallback() {
	fallbackCoverImageUUID := "definitely not a valid preset cover image ID"
	event := &Event{
		CoverImageUUID:         nil,
		FallbackCoverImageUUID: &fallbackCoverImageUUID,
	}

	url := event.CoverImageSourceURL()

	assert.Nil(suite.T(), url)
}

type CoverImageURLTemplateSuite struct {
	suite.Suite
}

func TestCoverImageURLTemplateSuite(t *testing.T) {
	suite.Run(t, &CoverImageURLTemplateSuite{})
}

func (suite *CoverImageURLTemplateSuite) SetupTest() {
	config.Values.CDNCoverImagePrefix = "some-cdn-prefix"
}

func (suite *CoverImageURLTemplateSuite) TestCoverImageURLTemplate_Nil() {
	event := &Event{
		CoverImageUUID:         nil,
		FallbackCoverImageUUID: nil,
	}

	assert.Nil(suite.T(), event.CoverImageURLTemplate())
}

func (suite *CoverImageURLTemplateSuite) TestCoverImageURLTemplate_Custom() {
	coverImageUUID := "randomly-generated-uuid"
	event := &Event{
		ChannelID:              123,
		CoverImageUUID:         &coverImageUUID,
		FallbackCoverImageUUID: nil,
	}

	url := event.CoverImageURLTemplate()

	if assert.NotNil(suite.T(), url) {
		assert.Equal(suite.T(), "https://static-cdn.jtvnw.net/some-cdn-prefix/123/randomly-generated-uuid-{width}x{height}", *url)
	}
}

func (suite *CoverImageURLTemplateSuite) TestCoverImageURLTemplate_Fallback() {
	fallbackCoverImageUUID := "city"
	event := &Event{
		CoverImageUUID:         nil,
		FallbackCoverImageUUID: &fallbackCoverImageUUID,
	}

	url := event.CoverImageURLTemplate()

	if assert.NotNil(suite.T(), url) {
		assert.Equal(suite.T(), "https://static-cdn.jtvnw.net/some-cdn-prefix/defaults/city-{width}x{height}", *url)
	}
}

func (suite *CoverImageURLTemplateSuite) TestCoverImageURLTemplate_InvalidFallback() {
	fallbackCoverImageUUID := "definitely not a valid preset cover image ID"
	event := &Event{
		CoverImageUUID:         nil,
		FallbackCoverImageUUID: &fallbackCoverImageUUID,
	}

	url := event.CoverImageURLTemplate()

	assert.Nil(suite.T(), url)
}

func TestURL(t *testing.T) {
	event := &Event{
		ID: 123,
	}

	assert.Equal(t, "https://www.twitch.tv/events/123", event.URL())
}
