package db

import (
	"context"
	"time"
)

// DB is the generic database interface, to be implemented by
// a specific database package.
type DB interface {
	// EVENTS

	// InsertEvent makes an INSERT query to persist an Event.
	InsertEvent(ctx context.Context, params *Event) (*Event, error)

	// SelectEventByID queries for a single Event by ID.
	SelectEventByID(ctx context.Context, id int) (*Event, error)

	// UpdateEvent makes an UPDATE query to persistent changes to an Event.
	UpdateEvent(ctx context.Context, params *Event) (*Event, error)

	// UpdateEventStatus changes the status of an event object to the correct status
	UpdateEventStatus(ctx context.Context, id int, status string) (*Event, error)

	// SelectAvailableEventsByChannelID queries for a list of Events by ChannelID.
	SelectAvailableEventsByChannelID(ctx context.Context, params *EventListParams) ([]*Event, error)

	// Select All events by start_date within the specified time range.
	SelectAllEventsWithinTimeRange(ctx context.Context, startTime time.Time, endTime time.Time) ([]*Event, error)

	// Select events for the specified game by start_date within the specified time range ordered by subscription count.
	SelectGameEventsWithinTimeRange(ctx context.Context, startTime time.Time, endTime time.Time, gameID int, params *EventSelectionParams) (*EventSelectionResponse, error)

	// Select events for the specified game by start_date within the specified time range that include the specified extension
	SelectGameEventsWithinTimeRangeWithExtension(ctx context.Context, startTime time.Time, endTime time.Time, gameID int, extensionField string, params *EventSelectionParams) (*EventSelectionResponse, error)

	// SelectLiveEventsByGameForTime
	SelectLiveEventsByGameForTime(ctx context.Context, liveTime time.Time, gameID int, params *EventSelectionParams) (*EventSelectionResponse, error)

	// ExistingCoverImage returns true if an event record with the given cover image UUID exists.
	ExistingCoverImage(ctx context.Context, coverImageID string) (bool, error)

	// EVENT EXTENSIONS
	// InsertEventExtension makes an INSERT query to persist an EventExtension.
	InsertEventExtension(ctx context.Context, params *EventExtension) (*EventExtension, error)

	// UpdateEventExtension makes an UPDATE query to persistent changes to an EventExtension.
	UpdateEventExtension(ctx context.Context, params *EventExtension) (*EventExtension, error)

	// DeleteEventExtension makes an DELETE query on EventExtension.
	DeleteEventExtensionByEventIDAndKey(ctx context.Context, eventID int, key string) error

	// SelectEventExtensionsByEventID queries for a single EventExtension by Event ID.
	SelectEventExtensionsByEventID(ctx context.Context, eventID int) ([]*EventExtension, error)
	// SelectEventExtensionsByEventIDs queries for multiple EventExtension by Event IDs.
	SelectEventExtensionsByEventIDs(ctx context.Context, eventIDs []int) ([]*EventExtension, error)
	// SelectEventExtensionByEventIDAndKey queries for a single Event by ID and key.
	SelectEventExtensionByEventIDAndKey(ctx context.Context, eventID int, key string) (*EventExtension, error)

	// EVENT STATS

	// IncrementEventEmailNotificationCount adds 1 to the existing count
	// of users with email notification enabled for a given event.
	//
	// On success, the updated count is returned.
	IncrementEventEmailNotificationCount(ctx context.Context, id int) (int, error)

	// DecrementEventEmailNotificationCount subtracts 1 from the existing count
	// of users with email notification enabled for a given event.
	//
	// On success, the updated count is returned.
	DecrementEventEmailNotificationCount(ctx context.Context, id int) (int, error)

	// FEATURED CHANNELS

	// CreateFeautredChannel marks a channel as featured, meaning we should include their content in featured lists
	CreateFeaturedChannel(ctx context.Context, id int) (int, error)

	// DeleteFeaturedChannel  unmarks a channel as featured, their content will no longer be included in featured lists
	DeleteFeaturedChannel(ctx context.Context, id int) (int, error)

	// ListFeauturedChannels enumerates all currently featured channels
	ListFeaturedChannels(ctx context.Context) ([]*FeaturedChannel, error)

	// MANAGER EVENT WITH EVENT STATS

	// SelectAvailableManagerEventsByChannelID makes a SELECT query from the
	// `events` table, joined by the `event_stats` table, for a list of event
	// records.
	//
	// The query is filtered by the passed-in EventListParams.
	SelectAvailableManagerEventsByChannelID(ctx context.Context, params *EventListParams) ([]*ManagerEvent, error)
}
