package pq

import (
	"context"
	"database/sql"
	"fmt"
	"strings"

	model "code.justin.tv/cb/oracle/internal/clients/db"
	log "github.com/Sirupsen/logrus"
)

// InsertEventExtension makes an INSERT query to the `event_extensions` table for a specific event ID.
func (db *DB) InsertEventExtension(ctx context.Context, params *model.EventExtension) (*model.EventExtension, error) {
	statement := `
	  INSERT INTO event_extensions (event_id, key, value)
		VALUES ($1, $2, $3)
		RETURNING event_id, key, value
	`

	eventExtension := &model.EventExtension{}

	row := db.QueryRowContext(ctx, statement, params.EventID, params.Key, params.Value)
	err := row.Scan(
		&eventExtension.EventID,
		&eventExtension.Key,
		&eventExtension.Value,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		log.WithError(err).WithField("event_extension", *params).Error("pq: failed to insert event extension")

		return eventExtension, err
	}

	return eventExtension, nil
}

// UpdateEventExtension makes an UPDATE query from the `event_extensions` table
// for a specific event record by ID.
func (db *DB) UpdateEventExtension(ctx context.Context, params *model.EventExtension) (*model.EventExtension, error) {
	statement := `
		UPDATE event_extensions
		SET value = $1
		WHERE event_id = $2 AND key = $3
		RETURNING event_id, key, value
	`

	eventExtension := &model.EventExtension{}

	row := db.QueryRowContext(ctx, statement, params.Value, params.EventID, params.Key)
	err := row.Scan(
		&eventExtension.EventID,
		&eventExtension.Key,
		&eventExtension.Value,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		log.WithError(err).WithField("event", *params).Error("pq: failed to update event extension")
		return nil, err
	}

	return eventExtension, nil
}

// SelectEventExtensionByEventIDAndKey makes a SELECT query from the `event_extensions` table.
func (db *DB) SelectEventExtensionByEventIDAndKey(ctx context.Context, eventID int, key string) (*model.EventExtension, error) {
	statement := `
		SELECT event_id, key, value
		FROM event_extensions
		WHERE event_id = $1 AND key = $2
		LIMIT 1
	`

	eventExtension := &model.EventExtension{}

	row := db.QueryRowContext(ctx, statement, eventID, key)
	err := row.Scan(
		&eventExtension.EventID,
		&eventExtension.Key,
		&eventExtension.Value,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		log.WithError(err).WithFields(log.Fields{
			"event_id": eventID,
			"key":      key,
		}).Error("pq: failed to select event extension by ID")

		return nil, err
	}

	return eventExtension, nil
}

// SelectEventExtensionsByEventIDs queries for multiple EventExtension by Event IDs.
func (db *DB) SelectEventExtensionsByEventIDs(ctx context.Context, eventIDs []int) ([]*model.EventExtension, error) {
	var eventExtensions []*model.EventExtension

	if len(eventIDs) == 0 {
		return eventExtensions, nil
	}

	template := `
		SELECT event_id, key, value
		FROM event_extensions
		WHERE event_id IN (%s)
		ORDER BY event_id
	`

	commaSeparatedIDs := strings.Trim(strings.Replace(fmt.Sprint(eventIDs), " ", ",", -1), "[]")
	statement := fmt.Sprintf(template, commaSeparatedIDs)

	rows, err := db.QueryContext(ctx, statement)
	if err != nil {
		msg := "pq: failed to select event extensions by event ids"
		log.WithError(err).WithField("event_ids", eventIDs).Error(msg)

		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "pq: failed to close rows when selecting event extensions by event ids"
			log.WithError(err).WithField("event_ids", eventIDs).Error(msg)
		}
	}()

	for rows.Next() {
		eventExtension := &model.EventExtension{}
		err = rows.Scan(
			&eventExtension.EventID,
			&eventExtension.Key,
			&eventExtension.Value,
		)

		eventExtensions = append(eventExtensions, eventExtension)
	}

	err = rows.Err()
	if err != nil {
		msg := "pq: error scanning rows when selecting event extensions by event ids"
		log.WithError(err).WithField("event_ids", eventIDs).Error(msg)

		return nil, err
	}

	return eventExtensions, nil
}

// SelectEventExtensionsByEventID makes a SELECT query from the `event_extensions` table.
func (db *DB) SelectEventExtensionsByEventID(ctx context.Context, eventID int) ([]*model.EventExtension, error) {
	statement := `
		SELECT event_id, key, value
		FROM event_extensions
		WHERE event_id = $1
	`

	var eventExtensions []*model.EventExtension
	contextLogger := log.WithField("event_id", eventID)

	rows, err := db.QueryContext(ctx, statement, eventID)
	if err != nil {
		contextLogger.WithError(err).Error("pq: failed to select available event extensions by event ID")

		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "pq: failed to close rows when selecting available event extensions by event ID"
			contextLogger.WithError(err).Error(msg)
		}
	}()

	for rows.Next() {
		eventExtension := &model.EventExtension{}
		err = rows.Scan(
			&eventExtension.EventID,
			&eventExtension.Key,
			&eventExtension.Value,
		)

		eventExtensions = append(eventExtensions, eventExtension)
	}

	err = rows.Err()
	if err != nil {
		msg := "pq: error scanning rows when selecting available event extensions by event ID"
		contextLogger.WithError(err).Error(msg)

		return nil, err
	}

	return eventExtensions, nil
}

// DeleteEventExtensionByEventIDAndKey(ctx context.Context, eventID int, key string) error
func (db *DB) DeleteEventExtensionByEventIDAndKey(ctx context.Context, eventID int, key string) error {
	statement := `
		DELETE
		FROM event_extensions
		WHERE event_id = $1 AND key = $2
	`

	_, err := db.ExecContext(ctx, statement, eventID, key)

	switch {
	case err == sql.ErrNoRows:
		return nil
	case err != nil:
		log.WithError(err).WithFields(log.Fields{
			"event_id": eventID,
			"key":      key,
		}).Error("pq: failed to select event extension by ID")

		return err
	}

	return nil
}
