package pq

import (
	"context"
	"database/sql"

	log "github.com/Sirupsen/logrus"
)

// IncrementEventEmailNotificationCount makes an UPDATE query to the
// `event_stats` table to add 1 to the `enabled_user_email_notification_count`
// column.
func (db *DB) IncrementEventEmailNotificationCount(ctx context.Context, eventID int) (int, error) {
	statement := `
    UPDATE event_stats
    SET enabled_user_email_notification_count = enabled_user_email_notification_count + 1
    WHERE event_id = $1
    RETURNING enabled_user_email_notification_count
	`

	var count int

	row := db.QueryRowContext(ctx, statement, eventID)
	err := row.Scan(&count)

	if err != nil {
		logger := log.WithError(err).WithField("event_id", eventID)

		if err == sql.ErrNoRows {
			logger.Warn("pq: incrementing 'event_stats.enabled_user_email_notification_count' resulted in no rows updated")
		} else {
			logger.Error("pq: failed to update event_stats")
		}

		return 0, err
	}

	return count, nil
}

// DecrementEventEmailNotificationCount makes an UPDATE query to the
// `event_stats` table to subtract 1 from the `enabled_user_email_notification_count`
// column.
func (db *DB) DecrementEventEmailNotificationCount(ctx context.Context, eventID int) (int, error) {
	statement := `
    UPDATE event_stats
    SET enabled_user_email_notification_count = enabled_user_email_notification_count - 1
    WHERE event_id = $1
    RETURNING enabled_user_email_notification_count
	`

	var count int

	row := db.QueryRowContext(ctx, statement, eventID)
	err := row.Scan(&count)

	if err != nil {
		logger := log.WithError(err).WithField("event_id", eventID)

		if err == sql.ErrNoRows {
			logger.Warn("pq: decrementing 'event_stats.enabled_user_email_notification_count' resulted in no rows updated")
		} else {
			logger.Error("pq: failed to update event_stats")
		}

		return 0, err
	}

	return count, nil
}
