package pq

import (
	"context"
	"database/sql"
	"fmt"

	"time"

	model "code.justin.tv/cb/oracle/internal/clients/db"
	log "github.com/Sirupsen/logrus"
)

// InsertEvent makes an INSERT query to the `events` table for a specific channel ID.
func (db *DB) InsertEvent(ctx context.Context, eventParams *model.Event) (*model.Event, error) {
	statement := `
	  INSERT INTO events (
			channel_id,
			start_time_utc,
			end_time_utc,
			time_zone_id,
			title,
			game_id,
			description,
			cover_image_uuid,
			fallback_cover_image_uuid
		)
		VALUES (
			$1, $2, $3, $4, $5, $6, $7, $8, $9
		)
		RETURNING
			id, channel_id, start_time_utc, end_time_utc, time_zone_id, title,
			game_id, description, cover_image_uuid, fallback_cover_image_uuid,
			status, created_at_utc
		`

	event := &model.Event{}

	row := db.QueryRowContext(
		ctx,
		statement,
		eventParams.ChannelID,
		eventParams.StartTimeUTC.UTC(),
		eventParams.EndTimeUTC.UTC(),
		eventParams.TimeZoneID,
		eventParams.Title,
		eventParams.GameID,
		eventParams.Description,
		eventParams.CoverImageUUID,
		eventParams.FallbackCoverImageUUID,
	)
	err := row.Scan(
		&event.ID,
		&event.ChannelID,
		&event.StartTimeUTC,
		&event.EndTimeUTC,
		&event.TimeZoneID,
		&event.Title,
		&event.GameID,
		&event.Description,
		&event.CoverImageUUID,
		&event.FallbackCoverImageUUID,
		&event.Status,
		&event.CreatedAtUTC,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		log.WithError(err).WithField("event", *eventParams).Error("pq: failed to insert event")

		return event, err
	}

	return event, nil
}

// SelectEventByID makes a SELECT query from the `events` table
// for a specific event record by ID, and joins the `cover_images`
// table for the associated cover image record with an 'available'
// status.
func (db *DB) SelectEventByID(ctx context.Context, id int) (*model.Event, error) {
	statement := `
		SELECT
			id, channel_id, start_time_utc, end_time_utc, time_zone_id, title,
			game_id, description, status, created_at_utc, updated_at_utc,
			cover_image_uuid, fallback_cover_image_uuid
		FROM events
		WHERE id = $1
	`

	event := &model.Event{}

	row := db.QueryRowContext(ctx, statement, id)
	err := row.Scan(
		&event.ID,
		&event.ChannelID,
		&event.StartTimeUTC,
		&event.EndTimeUTC,
		&event.TimeZoneID,
		&event.Title,
		&event.GameID,
		&event.Description,
		&event.Status,
		&event.CreatedAtUTC,
		&event.UpdatedAtUTC,
		&event.CoverImageUUID,
		&event.FallbackCoverImageUUID,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		log.WithError(err).WithField("id", id).Error("pq: failed to select event by ID")

		return nil, err
	}

	return event, nil
}

// UpdateEvent makes an UPDATE query from the `events` table
// for a specific event record by ID.
func (db *DB) UpdateEvent(ctx context.Context, eventParams *model.Event) (*model.Event, error) {
	statement := `
		UPDATE events
		SET
			start_time_utc = $1,
			end_time_utc = $2,
			time_zone_id = $3,
			title = $4,
			game_id = $5,
			description = $6,
			cover_image_uuid = $7
		WHERE id = $8
		RETURNING
			id, channel_id, start_time_utc, end_time_utc, time_zone_id, title,
			game_id, description, cover_image_uuid, fallback_cover_image_uuid,
			status, created_at_utc, updated_at_utc
	`

	event := &model.Event{}

	row := db.QueryRowContext(
		ctx,
		statement,
		eventParams.StartTimeUTC.UTC(),
		eventParams.EndTimeUTC.UTC(),
		eventParams.TimeZoneID,
		eventParams.Title,
		eventParams.GameID,
		eventParams.Description,
		eventParams.CoverImageUUID,
		eventParams.ID,
	)

	err := row.Scan(
		&event.ID,
		&event.ChannelID,
		&event.StartTimeUTC,
		&event.EndTimeUTC,
		&event.TimeZoneID,
		&event.Title,
		&event.GameID,
		&event.Description,
		&event.CoverImageUUID,
		&event.FallbackCoverImageUUID,
		&event.Status,
		&event.CreatedAtUTC,
		&event.UpdatedAtUTC,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		log.WithError(err).WithField("event", *eventParams).Error("pq: failed to update event")

		return nil, err
	}

	return event, nil
}

// UpdateEventStatus sets the event status to the specified status
//	this is used for deletes
func (db *DB) UpdateEventStatus(ctx context.Context, id int, status string) (*model.Event, error) {
	statement := `
		UPDATE events
		SET
			status = $1
		WHERE id = $2
		RETURNING
			id, channel_id, start_time_utc, end_time_utc, time_zone_id, title,
			game_id, description, cover_image_uuid, fallback_cover_image_uuid,
			status, created_at_utc, updated_at_utc
	`

	event := &model.Event{}

	row := db.QueryRowContext(ctx, statement, status, id)
	err := row.Scan(
		&event.ID,
		&event.ChannelID,
		&event.StartTimeUTC,
		&event.EndTimeUTC,
		&event.TimeZoneID,
		&event.Title,
		&event.GameID,
		&event.Description,
		&event.CoverImageUUID,
		&event.FallbackCoverImageUUID,
		&event.Status,
		&event.CreatedAtUTC,
		&event.UpdatedAtUTC,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		log.WithError(err).WithFields(log.Fields{
			"id":     id,
			"status": status,
		}).Error("pq: failed to update event status")

		return nil, err
	}

	return event, nil
}

// SelectAvailableEventsByChannelID makes a SELECT query from the `events` table
// for a specific event record by ID, and joins the `cover_images`
// table for the associated cover image record with an 'available'
// status. Sorts by start time.
func (db *DB) SelectAvailableEventsByChannelID(ctx context.Context, queryParams *model.EventListParams) ([]*model.Event, error) {
	statement := fmt.Sprintf(`
		SELECT
			id, channel_id, start_time_utc, end_time_utc, time_zone_id, title,
			game_id, description, cover_image_uuid, fallback_cover_image_uuid,
			status, created_at_utc, updated_at_utc
		FROM events
		WHERE channel_id = $1
			AND status = 'available'
			AND end_time_utc < $2
			AND end_time_utc > $3
		ORDER BY start_time_utc %v
		LIMIT $4
	`, queryParams.OrderBy)

	var events []*model.Event

	rows, err := db.QueryContext(
		ctx,
		statement,
		queryParams.ChannelID,
		queryParams.EndTimeBefore,
		queryParams.EndTimeAfter,
		model.SelectLimit,
	)

	contextLogger := log.WithFields(log.Fields{
		"channel_id":          queryParams.ChannelID,
		"end_time_before":     queryParams.EndTimeBefore,
		"end_time_after":      queryParams.EndTimeAfter,
		"start_time_order_by": queryParams.OrderBy,
	})

	if err != nil {
		contextLogger.WithError(err).Error("pq: failed to select available events by channel ID")

		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "pq: failed to close rows when selecting available events by channel ID"
			contextLogger.WithError(err).Error(msg)
		}
	}()

	for rows.Next() {
		event := &model.Event{}
		err = rows.Scan(
			&event.ID,
			&event.ChannelID,
			&event.StartTimeUTC,
			&event.EndTimeUTC,
			&event.TimeZoneID,
			&event.Title,
			&event.GameID,
			&event.Description,
			&event.CoverImageUUID,
			&event.FallbackCoverImageUUID,
			&event.Status,
			&event.CreatedAtUTC,
			&event.UpdatedAtUTC,
		)

		events = append(events, event)
	}

	err = rows.Err()
	if err != nil {
		msg := "pq: error scanning rows when selecting available events by channel ID"
		contextLogger.WithError(err).Error(msg)

		return nil, err
	}

	return events, nil
}

// Select All events by start_date within minute of provided time.
func (db *DB) SelectAllEventsWithinTimeRange(ctx context.Context, startTime time.Time, endTime time.Time) ([]*model.Event, error) {
	statement := `
		SELECT
			id, channel_id, start_time_utc, end_time_utc, time_zone_id,
			title, game_id, description, cover_image_uuid, fallback_cover_image_uuid, status, created_at_utc, updated_at_utc
		FROM events
		WHERE status = 'available'
			AND start_time_utc >= $1
			AND start_time_utc < $2
		ORDER BY start_time_utc
	`

	var events []*model.Event

	rows, err := db.QueryContext(ctx, statement, startTime, endTime)
	if err != nil {
		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).Error("Failed to close pg rows")
		}
	}()

	for rows.Next() {
		event := &model.Event{}
		err = rows.Scan(
			&event.ID,
			&event.ChannelID,
			&event.StartTimeUTC,
			&event.EndTimeUTC,
			&event.TimeZoneID,
			&event.Title,
			&event.GameID,
			&event.Description,
			&event.CoverImageUUID,
			&event.FallbackCoverImageUUID,
			&event.Status,
			&event.CreatedAtUTC,
			&event.UpdatedAtUTC,
		)

		events = append(events, event)
	}

	err = rows.Err()
	if err != nil {
		return nil, err
	}

	return events, nil
}

// Select events for a specified game by start_date within the specified time range
func (db *DB) SelectGameEventsWithinTimeRange(ctx context.Context, startTime time.Time, endTime time.Time, gameID int, params *model.EventSelectionParams) (*model.EventSelectionResponse, error) {

	limit := model.SelectLimit
	cursor := startTime
	if params != nil {
		if params.Limit != nil {
			limit = *params.Limit
		}
		if params.Cursor != nil {
			var err error
			cursor, err = time.Parse(time.RFC3339, *params.Cursor)
			if err != nil {
				log.WithError(err).Error("pg: failed to parse cursor")
				return nil, err
			}
		}
	}

	statement := `
		SELECT
      events.id, events.channel_id, events.start_time_utc, events.end_time_utc, events.time_zone_id,
      events.title, events.game_id, events.description, events.cover_image_uuid,
      events.fallback_cover_image_uuid, events.status, events.created_at_utc, events.updated_at_utc
		FROM events INNER JOIN event_stats ON events.id=event_stats.event_id
		WHERE events.status = 'available'
			AND events.start_time_utc >= $1
			AND events.start_time_utc < $2
			AND events.game_id = $3
		ORDER BY event_stats.enabled_user_email_notification_count DESC
		LIMIT $4
	`

	rows, err := db.Query(statement,
		cursor,
		endTime,
		gameID,
		limit,
	)

	if err != nil {
		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "Failed to close pg rows"
			log.WithError(err).Error(msg)
		}
	}()

	var events []*model.Event
	var lastStartTime time.Time
	for rows.Next() {
		event := &model.Event{}
		err = rows.Scan(
			&event.ID,
			&event.ChannelID,
			&event.StartTimeUTC,
			&event.EndTimeUTC,
			&event.TimeZoneID,
			&event.Title,
			&event.GameID,
			&event.Description,
			&event.CoverImageUUID,
			&event.FallbackCoverImageUUID,
			&event.Status,
			&event.CreatedAtUTC,
			&event.UpdatedAtUTC,
		)

		lastStartTime = event.StartTimeUTC
		events = append(events, event)
	}

	err = rows.Err()

	if err != nil {
		return nil, err
	}

	return &model.EventSelectionResponse{
		Events: events,
		Cursor: lastStartTime.Format(time.RFC3339),
	}, nil
}

func (db *DB) SelectGameEventsWithinTimeRangeWithExtension(ctx context.Context, startTime time.Time, endTime time.Time, gameID int, extensionField string, params *model.EventSelectionParams) (*model.EventSelectionResponse, error) {
	limit := model.SelectLimit
	cursor := startTime
	if params != nil {
		if params.Limit != nil {
			limit = *params.Limit
		}
		if params.Cursor != nil {
			var err error
			cursor, err = time.Parse(time.RFC3339, *params.Cursor)
			if err != nil {
				log.WithError(err).Error("pg: failed to parse cursor")
				return nil, err
			}
		}
	}

	statement := `
    SELECT
      events.id, events.channel_id, events.start_time_utc, events.end_time_utc, events.time_zone_id,
      events.title, events.game_id, events.description, events.cover_image_uuid,
      events.fallback_cover_image_uuid, events.status, events.created_at_utc, events.updated_at_utc,
      event_extensions.key, event_extensions.value
    FROM events INNER JOIN event_extensions ON events.id=event_extensions.event_id
    WHERE events.status = 'available'
      AND events.start_time_utc >= $1
      AND events.start_time_utc < $2
      AND events.game_id = $3
      AND event_extensions.key = $4
    ORDER BY start_time_utc
    LIMIT $5
  `

	rows, err := db.QueryContext(
		ctx,
		statement,
		cursor,
		endTime,
		gameID,
		extensionField,
		limit,
	)
	if err != nil {
		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).Error("Failed to close pg rows")
		}
	}()

	var events []*model.Event
	var lastStartTime time.Time
	for rows.Next() {
		event := &model.Event{}
		err = rows.Scan(
			&event.ID,
			&event.ChannelID,
			&event.StartTimeUTC,
			&event.EndTimeUTC,
			&event.TimeZoneID,
			&event.Title,
			&event.GameID,
			&event.Description,
			&event.CoverImageUUID,
			&event.FallbackCoverImageUUID,
			&event.Status,
			&event.CreatedAtUTC,
			&event.UpdatedAtUTC,
			&event.ExtensionField,
			&event.ExtensionValue,
		)

		lastStartTime = event.StartTimeUTC
		events = append(events, event)
	}

	err = rows.Err()

	if err != nil {
		return nil, err
	}

	return &model.EventSelectionResponse{
		Events: events,
		Cursor: lastStartTime.Format(time.RFC3339),
	}, nil
}

// Select events for a specified game by start_date that are running during the specified time
func (db *DB) SelectLiveEventsByGameForTime(ctx context.Context, liveTime time.Time, gameID int, params *model.EventSelectionParams) (*model.EventSelectionResponse, error) {
	limit := model.SelectLimit
	cursor := liveTime
	if params != nil {
		if params.Limit != nil {
			limit = *params.Limit
		}
		if params.Cursor != nil {
			var err error
			cursor, err = time.Parse(time.RFC3339, *params.Cursor)
			if err != nil {
				log.WithError(err).Error("pg: failed to parse cursor")
				return nil, err
			}
		}
	}

	statement := `
		SELECT
			id, channel_id, start_time_utc, end_time_utc, time_zone_id,
			title, game_id, description, cover_image_uuid, fallback_cover_image_uuid, status, created_at_utc, updated_at_utc
		FROM events
		WHERE status = 'available'
			AND start_time_utc < $1
      			AND end_time_utc > $2
      			AND game_id = $3
		ORDER BY start_time_utc
    		LIMIT $4
	`

	var events []*model.Event

	rows, err := db.QueryContext(ctx, statement, cursor, liveTime, gameID, limit)
	if err != nil {
		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).Error("Failed to close pg rows")
		}
	}()

	var lastStartTime time.Time
	for rows.Next() {
		event := &model.Event{}
		err = rows.Scan(
			&event.ID,
			&event.ChannelID,
			&event.StartTimeUTC,
			&event.EndTimeUTC,
			&event.TimeZoneID,
			&event.Title,
			&event.GameID,
			&event.Description,
			&event.CoverImageUUID,
			&event.FallbackCoverImageUUID,
			&event.Status,
			&event.CreatedAtUTC,
			&event.UpdatedAtUTC,
		)

		lastStartTime = event.StartTimeUTC
		events = append(events, event)
	}

	err = rows.Err()

	if err != nil {
		return nil, err
	}

	return &model.EventSelectionResponse{
		Events: events,
		Cursor: lastStartTime.Format(time.RFC3339),
	}, nil
}

// SelectAvailableManagerEventsByChannelID makes a SELECT query from the
// `events` table, joined by the `event_stats` table, for a list of event
// records.
//
// The query is filtered by the passed-in model.EventListParams.
func (db *DB) SelectAvailableManagerEventsByChannelID(ctx context.Context, params *model.EventListParams) ([]*model.ManagerEvent, error) {
	// COALESCE is used in the case that the corresponding `event_stats` row
	// does not exist and the selected `enabled_user_email_notification_count`
	// field is NULL.
	statement := fmt.Sprintf(`
		SELECT
			e.id, e.channel_id, e.start_time_utc, e.end_time_utc, e.time_zone_id,
			e.title, e.game_id, e.description, e.cover_image_uuid,
			e.fallback_cover_image_uuid, e.status, e.created_at_utc, e.updated_at_utc,
			COALESCE(stats.enabled_user_email_notification_count, 0)
		FROM events AS e
		LEFT JOIN event_stats AS stats
			ON stats.event_id = e.id
		WHERE e.channel_id = $1
			AND e.status = 'available'
			AND e.end_time_utc < $2
			AND e.end_time_utc > $3
		ORDER BY e.start_time_utc %v
		LIMIT $4
	`, params.OrderBy)

	var events []*model.ManagerEvent

	rows, err := db.QueryContext(
		ctx,
		statement,
		params.ChannelID,
		params.EndTimeBefore,
		params.EndTimeAfter,
		model.SelectLimit,
	)

	logger := log.WithFields(log.Fields{
		"channel_id":          params.ChannelID,
		"end_time_before":     params.EndTimeBefore,
		"end_time_after":      params.EndTimeAfter,
		"start_time_order_by": params.OrderBy,
	})

	if err != nil {
		logger.WithError(err).Error("pq: failed to select available manager events by channel ID")

		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "pq: failed to close rows when selecting available manager events by channel ID"
			logger.WithError(err).Error(msg)
		}
	}()

	for rows.Next() {
		event := &model.ManagerEvent{}
		err = rows.Scan(
			&event.ID,
			&event.ChannelID,
			&event.StartTimeUTC,
			&event.EndTimeUTC,
			&event.TimeZoneID,
			&event.Title,
			&event.GameID,
			&event.Description,
			&event.CoverImageUUID,
			&event.FallbackCoverImageUUID,
			&event.Status,
			&event.CreatedAtUTC,
			&event.UpdatedAtUTC,
			&event.EnabledUserEmailNotificationCount,
		)

		if err != nil {
			msg := "pq: error scanning rows when selecting available manager events by channel ID"
			logger.WithError(err).Error(msg)

			return nil, err
		}

		events = append(events, event)
	}

	err = rows.Err()
	if err != nil {
		msg := "pq: error during row iteration when selecting available manager events by channel ID"
		logger.WithError(err).Error(msg)

		return nil, err
	}

	return events, nil
}

// ExistingCoverImage makes a SELECT query from the `events` table
// for a specific event record with the given `cover_image_uuid`.
//
// Returning `true` means that a record with the given `cover_image_uuid`
// exists in the database.
func (db *DB) ExistingCoverImage(ctx context.Context, id string) (bool, error) {
	statement := `
		SELECT id
		FROM events
		WHERE cover_image_uuid = $1
	`

	event := &model.Event{}
	row := db.QueryRowContext(ctx, statement, id)
	err := row.Scan(&event.ID)

	switch {
	case err == sql.ErrNoRows:
		return false, nil
	case err != nil:
		log.WithError(err).WithFields(log.Fields{
			"events.cover_image_uuid": id,
		}).Error("pq: failed to select event by cover image ID")

		return false, err
	}

	return true, nil
}
