package pq

import (
	"context"
	"database/sql"

	model "code.justin.tv/cb/oracle/internal/clients/db"
	log "github.com/Sirupsen/logrus"
)

// ListFeaturedChannels SELECTs all featured channels that haven't been deleted
func (db *DB) ListFeaturedChannels(ctx context.Context) ([]*model.FeaturedChannel, error) {
	statement := `
    SELECT channel_id, created_at
    FROM featured_channels
    WHERE deleted_at IS NULL
  `

	var featured []*model.FeaturedChannel

	rows, err := db.QueryContext(ctx, statement)

	if err != nil {
		log.WithError(err).Error("pq: failed to select featured channels")
		return nil, err
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			msg := "pq: failed to close rows when selecting featured channels"
			log.WithError(err).Error(msg)
		}
	}()

	for rows.Next() {
		channel := &model.FeaturedChannel{}
		err = rows.Scan(
			&channel.ID,
			&channel.CreatedAtUTC,
		)
		if err != nil {
			continue
		}
		featured = append(featured, channel)
	}

	err = rows.Err()
	if err != nil {
		log.WithError(err).Error("pq: error scanning rows when selecting featured channels")
		return nil, err
	}

	return featured, nil
}

// CreateFeaturedChannel INSERTs a new channel entry or removes the deleted_at entry for that channel
func (db *DB) CreateFeaturedChannel(ctx context.Context, channelID int) (int, error) {
	statement := `
    INSERT INTO featured_channels (
      channel_id
    )
    VALUES (
      $1
    )
    ON CONFLICT (channel_id)
    DO UPDATE SET deleted_at = NULL
    RETURNING channel_id
  `

	var id int
	row := db.QueryRowContext(ctx, statement, channelID)
	err := row.Scan(&id)

	if err != nil {
		logger := log.WithError(err).WithField("channel_id", channelID)
		logger.Error("pq: failed to create featured_channel")
		return 0, err
	}

	return id, nil
}

// DeleteFeaturedChannel soft deletes the entry with a deleted entry
func (db *DB) DeleteFeaturedChannel(ctx context.Context, channelID int) (int, error) {
	statement := `
    UPDATE featured_channels
    SET deleted_at = current_timestamp
    WHERE channel_id = $1
    RETURNING channel_id
  `

	var id int
	row := db.QueryRowContext(ctx, statement, channelID)
	err := row.Scan(&id)

	if err != nil {
		logger := log.WithError(err).WithField("channel_id", channelID)
		if err == sql.ErrNoRows {
			logger.Warn("pq: deleting featured_channel resulted in no rows updated")
		} else {
			logger.Error("pq: failed to mark featured_channel deleted")
		}
		return 0, err
	}

	return id, nil
}
