package discovery

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	log "github.com/Sirupsen/logrus"
)

// HTTPClient is a wrapper for http.Client.
type HTTPClient struct {
	host   string
	client *http.Client
}

// NewHTTPClient instantiates and returns an HTTPClient.
func NewHTTPClient(host string) *HTTPClient {
	return &HTTPClient{
		host: host,
		client: &http.Client{
			Timeout: 1 * time.Second,
		},
	}
}

// GetGameByID calls on the Discovery service's GET /game?id= API.
func (h *HTTPClient) GetGameByID(ctx context.Context, id int) (*Game, error) {
	url := fmt.Sprint(h.host, "/game?id=", id)

	req, err := http.NewRequest(http.MethodGet, url, nil)
	if err != nil {
		return nil, err
	}

	contextLogger := log.WithFields(log.Fields{
		"method": req.Method,
		"url":    url,
	})

	req.Header.Set("Content-Type", "application/json")

	resp, err := h.client.Do(req)
	if err != nil {
		contextLogger.WithError(err).Error("discovery: failed to make HTTP request")
		return nil, err
	}

	defer func() {
		err = resp.Body.Close()
		if err != nil {
			contextLogger.WithError(err).Error("discovery: failed to close response body")
		}
	}()

	if resp.StatusCode != http.StatusOK {
		contextLogger.Warnf("discovery: received unexpected response with status %d", resp.StatusCode)
		return nil, &Error{Status: resp.StatusCode}
	}

	output := &Game{}

	err = json.NewDecoder(resp.Body).Decode(&output)
	if err != nil {
		contextLogger.WithError(err).Error("discovery: failed to decode response body to JSON")
		return nil, err
	}

	return output, nil
}

// GetGameListByIDs calls on the Discovery service's GET /games/list?id= API.
// id takes a comma separated list of ids
func (h *HTTPClient) GetGameListByIDs(ctx context.Context, gameIDs []int) (map[int]Game, error) {
	gamesByID := make(map[int]Game)
	if len(gameIDs) == 0 {
		return gamesByID, nil
	}

	// remove any duplicate IDs
	ids := make(map[int]bool)
	for _, gameID := range gameIDs {
		ids[gameID] = true
	}

	// create comma delimited list of game IDs to pass into discovery
	strIDs := []string{}
	for k := range ids {
		strIDs = append(strIDs, strconv.Itoa(k))
	}
	discoveryParams := strings.Join(strIDs, ",")

	// ping discovery for list of games IDs
	url := fmt.Sprint(h.host, "/game/list?id=", discoveryParams)
	req, err := http.NewRequest(http.MethodGet, url, nil)
	if err != nil {
		return nil, err
	}

	contextLogger := log.WithFields(log.Fields{
		"method": req.Method,
		"url":    url,
	})

	req.Header.Set("Content-Type", "application/json")

	resp, err := h.client.Do(req)
	if err != nil {
		contextLogger.WithError(err).Error("discovery: failed to make HTTP request")
		return nil, err
	}

	defer func() {
		err = resp.Body.Close()
		if err != nil {
			contextLogger.WithError(err).Error("discovery: failed to close response body")
		}
	}()

	if resp.StatusCode != http.StatusOK {
		contextLogger.Warnf("discovery: received unexpected response with status %d", resp.StatusCode)
		return nil, &Error{Status: resp.StatusCode}
	}

	var output []*Game

	err = json.NewDecoder(resp.Body).Decode(&output)
	if err != nil {
		contextLogger.WithError(err).Error("discovery: failed to decode response body to JSON")
		return nil, err
	}

	// key each game by its ID
	for _, game := range output {
		gamesByID[game.ID] = *game
	}

	return gamesByID, nil
}
