package discovery_test

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/cb/oracle/internal/clients/discovery"
)

func TestNewHTTPClient(t *testing.T) {
	client := discovery.NewHTTPClient("some-host")

	assert.NotNil(t, client)
}

func TestGetGameByID_InvalidHost(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusBadRequest)

		payload := `
			{
				"status": 400
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client := discovery.NewHTTPClient("invalid-host")

	if a.NotNil(client) {
		resp, err := client.GetGameByID(context.Background(), 123)
		a.Error(err)
		a.Contains(err.Error(), "unsupported protocol scheme")
		a.Nil(resp)
	}
}

func TestGetGameByID_Failure(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusBadRequest)

		payload := `
			{
				"status": 400
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client := discovery.NewHTTPClient(server.URL)

	if a.NotNil(client) {
		resp, err := client.GetGameByID(context.Background(), 123)
		a.Error(err)
		a.IsType(&discovery.Error{}, err)
		a.Contains(err.Error(), http.StatusText(http.StatusBadRequest))
		a.Nil(resp)
	}
}

func TestGetGameByID_MalformedResponse(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte("this is not valid JSON"))
		a.NoError(err)
	}))

	defer server.Close()

	client := discovery.NewHTTPClient(server.URL)

	if a.NotNil(client) {
		resp, err := client.GetGameByID(context.Background(), 123)
		a.Error(err)
		a.IsType(&json.SyntaxError{}, err)
		a.Nil(resp)
	}
}

func TestGetGameByID_Success(t *testing.T) {
	a := assert.New(t)

	gameID := 123
	gameName := "Legend of Testa"
	largeBoxArt := "http://large.box.art"
	mediumBoxArt := "http://medium.box.art"
	smallBoxArt := "http://small.box.art"

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		a.Equal(fmt.Sprintf("/game?id=%d", gameID), r.URL.RequestURI())
		a.Equal(http.MethodGet, r.Method)

		w.WriteHeader(http.StatusOK)

		payloadFmt := `
			{
				"id": %d,
				"name": "%s",
				"boxArt": {
					"large": "%s",
					"medium": "%s",
					"small": "%s"
				}
			}
		`

		payload := fmt.Sprintf(
			payloadFmt,
			gameID,
			gameName,
			largeBoxArt,
			mediumBoxArt,
			smallBoxArt,
		)

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client := discovery.NewHTTPClient(server.URL)

	if a.NotNil(client) {
		resp, err := client.GetGameByID(context.Background(), gameID)
		a.NoError(err)

		if a.NotNil(resp) {
			a.Equal(gameID, resp.ID)
			a.Equal(gameName, resp.Name)

			a.Equal(largeBoxArt, resp.BoxArt.Large)
			a.Equal(mediumBoxArt, resp.BoxArt.Medium)
			a.Equal(smallBoxArt, resp.BoxArt.Small)
		}
	}
}

func TestGetGameListByID_EmptyList_Success(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
	}))

	defer server.Close()

	client := discovery.NewHTTPClient(server.URL)

	if a.NotNil(client) {
		resp, err := client.GetGameListByIDs(context.Background(), []int{})
		a.Equal(0, len(resp))
		a.NoError(err)
	}
}
