package dynamodb

import (
	"context"
	"errors"
	"strconv"

	"code.justin.tv/cb/oracle/config"
	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	awsDynamoDB "github.com/aws/aws-sdk-go/service/dynamodb"
)

// ErrNoQueryItems is used when dynamodb.DyanamoDB.Query() returns a QueryOuput
// containing no items.
var ErrNoQueryItems = errors.New("dynamodb: query returned no items")

// DynamoDB is a wrapper for the aws-sdk-go/dynamo client.
type DynamoDB struct {
	Client    *awsDynamoDB.DynamoDB
	TableName *string
}

type SubscriptionItem struct {
	UserID  string
	EventID int
	Email   bool
}

func NewDynamoClient(environment string) (*DynamoDB, error) {
	tableName := config.Values.DynamoDBTableName

	log.Info("Starting session for DynamoDB at ", tableName)

	sess, err := session.NewSession()
	if err != nil {
		return nil, err
	}

	awsConfig := &aws.Config{
		Region: aws.String(config.Values.DynamoDBRegion),
	}

	if environment == "development" {
		c := credentials.NewSharedCredentials("", config.Values.AWSAccountProfile)
		awsConfig = awsConfig.WithCredentials(c)
	}

	svc := awsDynamoDB.New(sess, awsConfig)

	return &DynamoDB{
		Client:    svc,
		TableName: &tableName,
	}, nil
}

// TODO: need to paginate on last evaluated key
func (d *DynamoDB) GetUserNotificationsForEvent(ctx context.Context, eventID int, lastEvaluated map[string]*awsDynamoDB.AttributeValue) ([]*SubscriptionItem, map[string]*awsDynamoDB.AttributeValue, error) {
	dynamoQueryItem := &awsDynamoDB.QueryInput{
		TableName:              d.TableName,
		KeyConditionExpression: aws.String("EventID = :eventID"),

		ExpressionAttributeValues: map[string]*awsDynamoDB.AttributeValue{
			":eventID": {
				N: aws.String(strconv.Itoa(eventID)),
			},
		},
	}

	err := dynamoQueryItem.Validate()
	if err != nil {
		return nil, nil, err
	}

	queryResult, err := d.Client.Query(dynamoQueryItem)
	if err != nil {
		log.WithError(err).WithField("input", dynamoQueryItem.String()).Error("dynamodb: failed to query")

		return nil, nil, err
	}

	subscriptions := []*SubscriptionItem{}
	for _, item := range queryResult.Items {
		if item["UserID"].N != nil {
			s := &SubscriptionItem{
				UserID:  *item["UserID"].N,
				EventID: eventID,
			}
			subscriptions = append(subscriptions, s)
		}
	}

	return subscriptions, queryResult.LastEvaluatedKey, nil
}

func (d *DynamoDB) GetEventNotificationsForUser(ctx context.Context, eventID int, userID string) (*SubscriptionItem, error) {
	dynamoQueryItem := &awsDynamoDB.QueryInput{
		TableName:              d.TableName,
		KeyConditionExpression: aws.String("EventID = :eventID AND UserID = :userID"),

		ExpressionAttributeValues: map[string]*awsDynamoDB.AttributeValue{
			":eventID": {
				N: aws.String(strconv.Itoa(eventID)),
			},
			":userID": {
				N: &userID,
			},
		},
	}

	err := dynamoQueryItem.Validate()
	if err != nil {
		return nil, err
	}

	queryResult, err := d.Client.Query(dynamoQueryItem)
	if err != nil {
		log.WithError(err).WithField("input", dynamoQueryItem.String()).Error("dynamodb: failed to query")
		return nil, err
	}

	if aws.Int64Value(queryResult.Count) == 0 {
		return nil, ErrNoQueryItems
	}

	subscriptionItem := &SubscriptionItem{
		UserID:  userID,
		EventID: eventID,
		Email:   aws.BoolValue(queryResult.Items[0]["Email"].BOOL),
	}

	return subscriptionItem, nil
}

func (d *DynamoDB) GetEventsNotificationsForUser(ctx context.Context, eventIDs []int, userID string) ([]*SubscriptionItem, error) {
	items := make([]*SubscriptionItem, 0, len(eventIDs))
	if len(eventIDs) == 0 {
		return items, nil
	}
	keys := make([]map[string]*awsDynamoDB.AttributeValue, 0, len(eventIDs))
	for _, eventID := range eventIDs {
		keys = append(keys, map[string]*awsDynamoDB.AttributeValue{
			"EventID": {N: aws.String(strconv.Itoa(eventID))},
			"UserID":  {N: aws.String(userID)},
		})
	}
	input := &awsDynamoDB.BatchGetItemInput{
		RequestItems: map[string]*awsDynamoDB.KeysAndAttributes{
			*d.TableName: {Keys: keys},
		},
	}

	err := input.Validate()
	if err != nil {
		return nil, err
	}

	output, err := d.Client.BatchGetItem(input)
	if err != nil {
		log.WithError(err).WithField("input", input.String()).Error("dynamodb: failed to query")
		return nil, err
	}

	if len(output.Responses[*d.TableName]) == 0 {
		return nil, ErrNoQueryItems
	}

	for _, response := range output.Responses[*d.TableName] {
		eventID, err := strconv.Atoi(*response["EventID"].N)
		if err != nil {
			log.WithError(err).WithField("output", output.String()).Error("dynamodb: EventID not a number")
			return nil, err
		}
		item := &SubscriptionItem{
			UserID:  *response["UserID"].N,
			EventID: eventID,
			Email:   aws.BoolValue(response["Email"].BOOL),
		}
		items = append(items, item)
	}

	return items, nil
}

func (d *DynamoDB) SubscribeEvent(ctx context.Context, params SubscriptionItem) error {
	notificationTypes := []*string{}
	if params.Email {
		s := "email"
		notificationTypes = append(notificationTypes, &s)
	}

	dynamoPutItem := &awsDynamoDB.PutItemInput{
		TableName: d.TableName,
		Item: map[string]*awsDynamoDB.AttributeValue{
			"EventID": {
				N: aws.String(strconv.Itoa(params.EventID)),
			},
			"UserID": {
				N: &params.UserID,
			},
			"Email": {
				BOOL: aws.Bool(true),
			},
		},
	}

	err := dynamoPutItem.Validate()
	if err != nil {
		return err
	}

	_, err = d.Client.PutItem(dynamoPutItem)
	if err != nil {
		log.WithError(err).WithField("input", dynamoPutItem.String()).Error("dynamodb: failed to put")

		return err
	}

	return nil
}

func (d *DynamoDB) UnsubscribeEvent(ctx context.Context, params SubscriptionItem) error {
	dynamoDeleteItem := &awsDynamoDB.DeleteItemInput{
		TableName: d.TableName,
		Key: map[string]*awsDynamoDB.AttributeValue{
			"EventID": {
				N: aws.String(strconv.Itoa(params.EventID)),
			},
			"UserID": {
				N: &params.UserID,
			},
		},
	}

	err := dynamoDeleteItem.Validate()
	if err != nil {
		return err
	}

	_, err = d.Client.DeleteItem(dynamoDeleteItem)
	if err != nil {
		log.WithError(err).WithField("input", dynamoDeleteItem.String()).Error("dynamodb: failed to delete")

		return err
	}

	return nil
}
