package s3

import (
	"bytes"
	"context"
	"encoding/base64"
	"errors"
	"fmt"
	"io"
	"net/http"

	"code.justin.tv/cb/oracle/config"

	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	awsS3Client "github.com/aws/aws-sdk-go/service/s3"
)

// The string constants below are the three different prefixes for
// object storage in the same S3 bucket.
const (
	PrefixImagesPending  = "images-pending"
	PrefixImagesSaved    = "images-saved"
	PrefixImagesExpiring = "images-expiring"
)

// S3 is a wrapper for the aws-sdk-go/s3 client.
type S3 struct {
	client *awsS3Client.S3
}

// CoverImage contains the data bytes as well as other meta data for
// an image file.
type CoverImage struct {
	Data          []byte
	Body          io.ReadSeeker
	ContentLength int64
	ContentType   string
}

// NewS3Client instantiates an S3 client.
func NewS3Client(environment string) (*S3, error) {
	log.Info("Starting session for S3 Client at ", config.Values.S3Region)

	sess, err := session.NewSession()
	if err != nil {
		return nil, err
	}

	awsConfig := &aws.Config{
		Region: aws.String(config.Values.S3Region),
	}

	if environment == "development" {
		c := credentials.NewSharedCredentials("", config.Values.AWSAccountProfile)
		awsConfig = awsConfig.WithCredentials(c)
	}

	return &S3{
		client: awsS3Client.New(sess, awsConfig),
	}, nil
}

// ExpireSavedImage copies an object in the 'saved' storage (PrefixImagesSaved)
// to the 'expiring' storage (PrefixImagesExpiring) with public read access.
//
// The original object in the 'saved' storage is set to be deleted within
// one day to remove it from public read access.
func (s *S3) ExpireSavedImage(ctx context.Context, channelID string, versionID string) error {
	bucket := config.Values.S3CoverImageBucket
	source := fmt.Sprintf("%s/%s/%s/%s", bucket, PrefixImagesSaved, channelID, versionID)
	dest := fmt.Sprintf("%s/%s/%s", PrefixImagesExpiring, channelID, versionID)

	// first we make a copy of this image so that we can access this in the future
	copyInp := &awsS3Client.CopyObjectInput{
		Bucket:     aws.String(bucket),
		Key:        aws.String(dest),
		ACL:        aws.String(awsS3Client.ObjectCannedACLPublicRead),
		CopySource: aws.String(source),
	}

	_, err := s.client.CopyObject(copyInp)
	if err != nil {
		log.WithField("input", copyInp.String()).WithError(err).Error("s3: failed to copy object")
		return err
	}

	// then we replace the old version with a delete marker (so the public cant access it)
	//	the ncv will get cleaned up by the lifecycle conf
	delInp := &awsS3Client.DeleteObjectInput{
		Bucket: aws.String(bucket),
		Key:    aws.String(source),
	}

	_, err = s.client.DeleteObject(delInp)
	if err != nil {
		log.WithField("input", delInp.String()).WithError(err).Error("s3: failed to delete object")
		return err
	}

	return nil
}

// ValidateAndUploadCoverImage validates a base64-encoded image and uploads
// the image object to the 'pending' storage (PrefixImagesPending).
func (s *S3) ValidateAndUploadCoverImage(ctx context.Context, channelID string, encodedImg string) (*string, error) {
	bucket := config.Values.S3CoverImageBucket
	dest := fmt.Sprintf("%s/%s", PrefixImagesPending, channelID)

	image, err := s.ValidateImage(ctx, encodedImg)
	if err != nil {
		return nil, err
	}

	params := &awsS3Client.PutObjectInput{
		Bucket:        aws.String(bucket),
		Key:           aws.String(dest),
		Body:          image.Body,
		ContentLength: aws.Int64(image.ContentLength),
		ContentType:   aws.String(image.ContentType),
	}

	resp, err := s.client.PutObject(params)

	if err != nil {
		log.WithField("input", params.String()).WithError(err).Error("s3: failed to put object")
		return nil, err
	} else if resp == nil {
		return nil, errors.New("s3: no response from S3")
	} else if resp.VersionId == nil {
		return nil, errors.New("s3: recevied null version from a versioned bucket")
	}

	return resp.VersionId, nil
}

// The constants below are error strings for validating an image sent with PostV1EventsInput.
const (
	InvalidEncoding    = "Could not decode image."
	InvalidContentType = "Image must be .jpg"
)

// ValidateImage validates a base 64 encdoed image to ensure its filetype and image size.
// Visage should be checking the max image
// TODO: does not check for max dimensions
func (s *S3) ValidateImage(ctx context.Context, encodedImg string) (*CoverImage, error) {
	data, err := base64.StdEncoding.DecodeString(encodedImg)
	if err != nil {
		return nil, errors.New(InvalidEncoding)
	}

	contentType := http.DetectContentType(data)

	if contentType != "image/jpeg" {
		return nil, errors.New(InvalidContentType)
	}

	image := &CoverImage{
		Data:          data,
		Body:          bytes.NewReader(data),
		ContentLength: int64(len(data)),
		ContentType:   contentType,
	}

	return image, nil
}

// SaveFromPending copies the existing object in the pending storage to
// a permanent 'saved' storage.
func (s *S3) SaveFromPending(ctx context.Context, channelID string, versionID string) error {
	bucket := config.Values.S3CoverImageBucket
	source := fmt.Sprintf("%s/%s/%s?versionId=%s", bucket, PrefixImagesPending, channelID, versionID)
	dest := fmt.Sprintf("%s/%s/%s", PrefixImagesSaved, channelID, versionID)

	inp := &awsS3Client.CopyObjectInput{
		Bucket:     aws.String(bucket),
		Key:        aws.String(dest),
		ACL:        aws.String(awsS3Client.ObjectCannedACLPublicRead),
		CopySource: aws.String(source),
	}

	_, err := s.client.CopyObject(inp)
	if err != nil {
		awsErr, ok := err.(awserr.Error)
		if !ok || awsErr.Code() != "InvalidArgument" {
			log.WithField("input", inp.String()).WithError(err).Error("s3: failed to copy object")
		}

		return err
	}

	return nil
}
