package sns

import (
	"encoding/json"
	"errors"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/cb/oracle/config"
	"code.justin.tv/cb/oracle/internal/clients/db"

	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
)

const (
	pushyEventStartName = "event_start"
	emailImageWidth     = 640
	emailImageHeight    = 360
)

// SNS is a wrapper for the aws-sdk-go/sns client.
type SNS struct {
	Client      *sns.SNS
	environment string
}

type NotificationType struct {
	Email  bool `json:"email"`
	Onsite bool `json:"onsite"`
	Mobile bool `json:"mobile"`
}

// NewS3Client instantiates an S3 client.
func NewSNSClient(environment string) (*SNS, error) {
	log.Info("Starting session for SNS Client at ", config.Values.S3Region)

	sess, err := session.NewSession()
	if err != nil {
		return nil, err
	}

	awsConfig := &aws.Config{
		Region: aws.String(config.Values.S3Region),
	}

	if environment == "development" {
		c := credentials.NewSharedCredentials("", config.Values.AWSAccountProfile)
		awsConfig = awsConfig.WithCredentials(c)
	}

	svc := sns.New(sess, awsConfig)

	return &SNS{
		Client:      svc,
		environment: config.Environment,
	}, nil
}

type NotificationMessage struct {
	EventID          string           `json:"event_id"`
	EventURL         string           `json:"event_url"`
	EventImage       *string          `json:"event_image"`
	EventTitle       string           `json:"event_title"`
	EventTime        string           `json:"event_time"`
	EventDescription *string          `json:"event_description"`
	EventGame        string           `json:"event_game"`
	ChannelID        string           `json:"channel_user_id"`
	UserID           string           `json:"user_id"`
	NotificationType NotificationType `json:"notification_type"`
}

// TODO: do a batch event publish in here so we don't have to pass in gameName
func (s *SNS) PublishEmailNotification(event *db.Event, notificationType NotificationType, userID int, gameName string) error {
	if event == nil {
		return errors.New("Event does not exist")
	}

	userIDStr := strconv.Itoa(userID)
	channelIDStr := strconv.Itoa(event.ChannelID)
	eventIDStr := strconv.Itoa(event.ID)

	coverImageURL := event.CoverImageURLTemplate()
	if coverImageURL != nil {
		*coverImageURL = strings.Replace(*coverImageURL, `{width}`, strconv.Itoa(emailImageWidth), -1)
		*coverImageURL = strings.Replace(*coverImageURL, `{height}`, strconv.Itoa(emailImageHeight), -1)
	}

	msg := &NotificationMessage{
		EventID:          eventIDStr,
		EventURL:         event.URL(),
		EventImage:       coverImageURL,
		EventTitle:       event.Title,
		EventTime:        event.StartTimeUTC.Format(time.RFC3339),
		EventDescription: event.Description,
		EventGame:        gameName,
		ChannelID:        channelIDStr,
		UserID:           userIDStr,
		NotificationType: notificationType,
	}

	message, err := json.Marshal(msg)
	if err != nil {
		return err
	}

	messageAttributes := map[string]*sns.MessageAttributeValue{
		"event": {
			DataType:    aws.String("String"),
			StringValue: aws.String(pushyEventStartName),
		},
	}

	publishInput := &sns.PublishInput{
		TopicArn:          aws.String(config.Values.SNSTopicArn),
		Message:           aws.String(string(message)),
		MessageAttributes: messageAttributes,
	}

	_, err = s.Client.Publish(publishInput)

	if err != nil {
		return err
	}
	return nil
}
