package tmi

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"regexp"
	"strings"
	"time"

	"code.justin.tv/cb/oracle/internal/clients/db"

	log "github.com/Sirupsen/logrus"
	gocache "github.com/patrickmn/go-cache"
)

const (
	clueServiceHost       = "http://tmi-clue.internal.twitch.tv"
	censorRegexCacheKey   = "banned_words"
	cacheTTL              = 2 * time.Hour
	cachePurgePeriod      = 30 * time.Second
	censorString          = `***`
	exactWordRegexPattern = `\b%s\b`
	prefixRegexPattern    = `\b%s(\S*)\b`
	postfixRegexPattern   = `\b(\S*)%s\b`
	infixRegexPattern     = `\b(\S*)%s(\S*)\b`
)

// TMIClient is a wrapper for the TMI HTTP client.
type TMIClient struct {
	host       string
	httpClient *http.Client
	cache      *gocache.Cache
}

// New instantiates and returns an TMIClient.
func New() (*TMIClient, error) {
	gocacheClient := gocache.New(cacheTTL, cachePurgePeriod)
	if gocacheClient == nil {
		return nil, errors.New("Error initializing gocache.")
	}
	return &TMIClient{
		host: clueServiceHost,
		httpClient: &http.Client{
			Timeout: 1 * time.Second,
		},
		cache: gocacheClient,
	}, nil
}

func (t *TMIClient) CensorEventInformation(ctx context.Context, event *db.Event) error {
	censorRegex, err := t.GetCensorRegex(ctx)
	if err != nil {
		return err
	}

	if event.Description != nil {
		d := censorRegex.ReplaceAllLiteralString(*event.Description, censorString)
		event.Description = &d
	}

	event.Title = censorRegex.ReplaceAllLiteralString(event.Title, censorString)

	return nil
}

func (t *TMIClient) GetCensorRegex(ctx context.Context) (*regexp.Regexp, error) {
	// first check go-cache to see if the banned words exists
	if censorRegex, found := t.cache.Get(censorRegexCacheKey); found {
		return censorRegex.(*regexp.Regexp), nil
	}

	bannedWords, err := t.GetGlobalBannedWords(ctx)
	if err != nil {
		return nil, err
	}

	censorTextRegex, err := CreateCensorRegexFromList(ctx, bannedWords.GlobalBannedWords)
	if err != nil {
		return nil, err
	}

	t.cache.Set(censorRegexCacheKey, censorTextRegex, cacheTTL)

	return censorTextRegex, nil
}

func (t *TMIClient) GetGlobalBannedWords(ctx context.Context) (*GlobalBannedWordsResponse, error) {
	url := fmt.Sprint(t.host, "/global/banned_words")

	req, err := http.NewRequest(http.MethodGet, url, nil)
	if err != nil {
		return nil, err
	}

	contextLogger := log.WithFields(log.Fields{
		"method": req.Method,
		"url":    url,
	})

	req.Header.Set("Content-Type", "application/json")

	resp, err := t.httpClient.Do(req)
	if err != nil {
		contextLogger.WithError(err).Error("tmi: failed to make HTTP request")
		return nil, err
	}

	defer func() {
		err = resp.Body.Close()
		if err != nil {
			contextLogger.WithError(err).Error("tmi: failed to close response body")
		}
	}()

	if resp.StatusCode != http.StatusOK {
		contextLogger.Warnf("tmi: received unexpected response with status %d", resp.StatusCode)
		return nil, &Error{Status: resp.StatusCode}
	}

	output := &GlobalBannedWordsResponse{}

	err = json.NewDecoder(resp.Body).Decode(&output)
	if err != nil {
		contextLogger.WithError(err).Error("tmi: failed to decode response body to JSON")
		return nil, err
	}

	return output, nil
}

// CreateCensorRegexFromList takes a list of global banned words and makes a combined regex rule with each word rule.
// Each banned word regex uses \b for word boundries to replace entire strings, if `*` are specified in the rule.
// Note that the GlobalBannedWord.Word string does not actually follow standard regex syntax.
// asdf will replace `asdf` but not `___asdf___` or `_asdf` or `asdf_`
// asdf* will replace `asdf_____` but not `___asdf___`
// *asdf will replace `_____asdf` but not `___asdf___`
// *asdf* takes the string `___asdf____` and turns it into `***`
// The combined regex uses `(?i)` to be case insensitive.
func CreateCensorRegexFromList(ctx context.Context, wordList []GlobalBannedWord) (*regexp.Regexp, error) {
	censorWordRegex := []string{}

	for _, bannedWord := range wordList {
		if bannedWord.CanOptOut == false {
			prefix := (string(bannedWord.Word[len(bannedWord.Word)-1]) == `*`)
			postfix := (string(bannedWord.Word[0]) == "*")

			noStars := strings.Trim(bannedWord.Word, `*`)
			var reString string
			if prefix && postfix {
				reString = fmt.Sprintf(infixRegexPattern, noStars)
			} else if prefix {
				reString = fmt.Sprintf(prefixRegexPattern, noStars)
			} else if postfix {
				reString = fmt.Sprintf(postfixRegexPattern, noStars)
			} else {
				reString = fmt.Sprintf(exactWordRegexPattern, noStars)
			}

			censorWordRegex = append(censorWordRegex, reString)
		}
	}

	return regexp.Compile("(?i)" + strings.Join(censorWordRegex, "|"))
}
