package tmi

import (
	"context"
	"net/http"
	"net/http/httptest"
	"regexp"
	"testing"

	"code.justin.tv/cb/oracle/internal/api/responder"
	"code.justin.tv/cb/oracle/internal/clients/db"

	gocache "github.com/patrickmn/go-cache"

	"github.com/stretchr/testify/assert"
)

func TestNewTMIClient(t *testing.T) {
	a := assert.New(t)

	client, err := New()
	a.NotNil(t, client)
	a.NoError(err)
}

func TestCreateCensorRegexFromList(t *testing.T) {
	a := assert.New(t)
	ctx := context.Background()

	wordList := []GlobalBannedWord{
		GlobalBannedWord{Word: "tacofucker", CanOptOut: false},
		GlobalBannedWord{Word: "pre*", CanOptOut: false},
		GlobalBannedWord{Word: "*post", CanOptOut: false},
		GlobalBannedWord{Word: "*inf*", CanOptOut: false},
		GlobalBannedWord{Word: "hi", CanOptOut: true},
	}

	compiledRegex, err := CreateCensorRegexFromList(ctx, wordList)
	a.NoError(err)

	expectedRegex := `(?i)\btacofucker\b|\bpre(\S*)\b|\b(\S*)post\b|\b(\S*)inf(\S*)\b`
	a.NoError(err)

	a.NotNil(t, compiledRegex)
	a.Equal(compiledRegex.String(), expectedRegex)
}

func TestGetCensorRegex(t *testing.T) {
	a := assert.New(t)
	ctx := context.Background()
	gocacheClient := gocache.New(cacheTTL, cachePurgePeriod)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		wordList := []GlobalBannedWord{
			GlobalBannedWord{Word: "tacofucker", CanOptOut: false},
			GlobalBannedWord{Word: "pre*", CanOptOut: false},
			GlobalBannedWord{Word: "*post", CanOptOut: false},
			GlobalBannedWord{Word: "*inf*", CanOptOut: false},
			GlobalBannedWord{Word: "hi", CanOptOut: true},
		}

		response := &GlobalBannedWordsResponse{
			GlobalBannedWords: wordList,
		}

		writer := responder.NewResponseWriter(w)
		writer.OK(response)
	}))

	defer server.Close()

	c := &TMIClient{
		cache:      gocacheClient,
		host:       server.URL,
		httpClient: &http.Client{},
	}

	// Verify that the new key does not yet exist in gocache
	cachedRegex, found := c.cache.Get(censorRegexCacheKey)
	a.Nil(cachedRegex)
	a.False(found)

	// Fetch banned words list
	re, err := c.GetCensorRegex(ctx)
	a.NoError(err)

	// Expect the regex to exclude `CanOptOut: true`
	expectedRegex := `(?i)\btacofucker\b|\bpre(\S*)\b|\b(\S*)post\b|\b(\S*)inf(\S*)\b`
	a.NoError(err)
	a.Equal(expectedRegex, re.String())

	// Verify that the new key exists in the gocache
	cachedRegex, found = c.cache.Get(censorRegexCacheKey)
	a.Equal(cachedRegex.(*regexp.Regexp).String(), expectedRegex)
	a.True(found)
}

func TestCensorEventInformation(t *testing.T) {
	a := assert.New(t)
	ctx := context.Background()
	gocacheClient := gocache.New(cacheTTL, cachePurgePeriod)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		wordList := []GlobalBannedWord{
			GlobalBannedWord{Word: "*tacofucker*", CanOptOut: false},
			GlobalBannedWord{Word: "prefix*", CanOptOut: false},
			GlobalBannedWord{Word: "*postfix", CanOptOut: false},
			GlobalBannedWord{Word: "badword", CanOptOut: false},
			GlobalBannedWord{Word: "hi", CanOptOut: true},
		}

		response := &GlobalBannedWordsResponse{
			GlobalBannedWords: wordList,
		}

		writer := responder.NewResponseWriter(w)
		writer.OK(response)
	}))

	defer server.Close()

	c := &TMIClient{
		cache:      gocacheClient,
		host:       server.URL,
		httpClient: &http.Client{},
	}

	description := "tacofucker _tacofucker tacofucker_ _tacofucker_ TACOFUCKER TACOFUCKERTACOFUCKER prefix _prefix prefix_ postfix _postfix postfix_ badword _badword badword_ hi mom"
	event := &db.Event{
		Title:       "tacofucker hello",
		Description: &description,
	}

	// Censor event information
	err := c.CensorEventInformation(ctx, event)
	a.NoError(err)

	// Expect to censor exact matches, prefix, suffix,
	expectedDescription := "*** *** *** *** *** *** *** _prefix *** *** *** postfix_ *** _badword badword_ hi mom"
	a.Equal(*event.Description, expectedDescription)
	a.Equal(event.Title, "*** hello")
}
