package tracking

import (
	"context"

	log "github.com/Sirupsen/logrus"

	"code.justin.tv/common/spade-client-go/spade"
)

// Client is a wrapper for the Spade client.
type Client struct {
	spade spade.Client
}

// NewClient makes a new tracking client!
func NewClient() (*Client, error) {
	client, err := spade.NewClient()
	if err != nil {
		return nil, err
	}

	return &Client{
		spade: client,
	}, nil
}

// SendServerActionEvent sends a tracking event to Spade,
// for when a user interacts with events (create, delete, or update).
//
// SendServerActionEvent is meant to be called in a goroutine.
func (c *Client) SendServerActionEvent(ctx context.Context, event *OracleServerEvent) {
	defer func() {
		if r := recover(); r != nil {
			log.WithField("recover", r).Errorf(
				"tracking: recovered from panic when sending event '%s'",
				OracleServerEventName,
			)
		}
	}()

	err := c.spade.TrackEvent(ctx, OracleServerEventName, *event)
	if err != nil {
		log.WithError(err).Errorf(
			"tracking: failed to send Spade event '%s'",
			OracleServerEventName,
		)
	}
}

// SendNotificationChangeEvent sends a tracking event to Spade,
// for when a user changes an event's notification setting.
//
// SendNotificationChangeEvent is meant to be called in a goroutine.
func (c *Client) SendNotificationChangeEvent(ctx context.Context, event *OracleNotificationEvent) {
	defer func() {
		if r := recover(); r != nil {
			log.WithField("recover", r).Errorf(
				"tracking: recovered from panic when sending event '%s'",
				OracleNotificationEventName,
			)
		}
	}()

	err := c.spade.TrackEvent(ctx, OracleNotificationEventName, *event)
	if err != nil {
		log.WithError(err).Errorf(
			"tracking: failed to send Spade event '%s'",
			OracleNotificationEventName,
		)
	}
}
