package users

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	log "github.com/Sirupsen/logrus"
)

// HTTPClient is a wrapper for http.Client.
type HTTPClient struct {
	host   string
	client *http.Client
}

// NewHTTPClient instantiates and returns an HTTPClient.
func NewHTTPClient(host string) *HTTPClient {
	return &HTTPClient{
		host: host,
		client: &http.Client{
			Timeout: 1 * time.Second,
		},
	}
}

// GetByID calls on the Users service's GET /users/:id API.
func (h *HTTPClient) GetByID(ctx context.Context, id int) (*User, error) {
	path := fmt.Sprintf("/users/%d", id)
	url := fmt.Sprint(h.host, path)

	req, err := http.NewRequest(http.MethodGet, url, nil)
	if err != nil {
		return nil, err
	}

	contextLogger := log.WithFields(log.Fields{
		"method": req.Method,
		"url":    url,
	})

	req.Header.Set("Content-Type", "application/json")

	resp, err := h.client.Do(req)
	if err != nil {
		contextLogger.WithError(err).Error("users: failed to make HTTP request")
		return nil, err
	}

	defer func() {
		err = resp.Body.Close()
		if err != nil {
			contextLogger.WithError(err).Error("users: failed to close response body")
		}
	}()

	if resp.StatusCode != http.StatusOK {
		contextLogger.Warnf("users: received unexpected response with status %d", resp.StatusCode)
		return nil, &Error{Status: resp.StatusCode}
	}

	output := &User{}

	err = json.NewDecoder(resp.Body).Decode(&output)
	if err != nil {
		contextLogger.WithError(err).Error("users: failed to decode response body to JSON")
		return nil, err
	}

	return output, nil
}
