package users_test

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/cb/oracle/internal/clients/users"
)

func TestNewHTTPClient(t *testing.T) {
	client := users.NewHTTPClient("some-host")

	assert.NotNil(t, client)
}

func TestGetByID_InvalidHost(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusBadRequest)

		payload := `
			{
				"status": 400
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client := users.NewHTTPClient("invalid-host")

	if a.NotNil(client) {
		resp, err := client.GetByID(context.Background(), 123)
		a.Error(err)
		a.Contains(err.Error(), "unsupported protocol scheme")
		a.Nil(resp)
	}
}

func TestGetByID_Failure(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusBadRequest)

		payload := `
			{
				"status": 400
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client := users.NewHTTPClient(server.URL)

	if a.NotNil(client) {
		resp, err := client.GetByID(context.Background(), 123)
		a.Error(err)
		a.IsType(&users.Error{}, err)
		a.Contains(err.Error(), http.StatusText(http.StatusBadRequest))
		a.Nil(resp)
	}
}

func TestGetByID_MalformedResponse(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte("this is not valid JSON"))
		a.NoError(err)
	}))

	defer server.Close()

	client := users.NewHTTPClient(server.URL)

	if a.NotNil(client) {
		resp, err := client.GetByID(context.Background(), 123)
		a.Error(err)
		a.IsType(&json.SyntaxError{}, err)
		a.Nil(resp)
	}
}

func TestGetByID_Success(t *testing.T) {
	a := assert.New(t)

	userID := 123

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		a.Equal(fmt.Sprintf("/users/%d", userID), r.URL.EscapedPath())
		a.Equal(http.MethodGet, r.Method)

		w.WriteHeader(http.StatusOK)

		payloadFmt := `
			{
				"id": %d
			}
		`

		payload := fmt.Sprintf(payloadFmt, userID)

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client := users.NewHTTPClient(server.URL)

	if a.NotNil(client) {
		resp, err := client.GetByID(context.Background(), userID)
		a.NoError(err)

		if a.NotNil(resp) {
			a.Equal(userID, resp.ID)
		}
	}
}
