package httpserver

import (
	"fmt"
	"net/http"
	"time"

	"goji.io"
	"goji.io/pat"

	log "github.com/Sirupsen/logrus"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/cb/oracle/internal/middleware"
)

type Server struct {
	*goji.Mux
	statter statsd.Statter
}

func NewServer(statter statsd.Statter) *Server {
	mux := goji.NewMux()
	mux.Use(middleware.PanicRecoverer)

	return &Server{
		Mux:     goji.NewMux(),
		statter: statter,
	}
}

// handlerFunc is a helper type declaration for the handle function.
type handlerFunc func(http.ResponseWriter, *http.Request)

// handle wraps each route-matching handler for logging as well as sending
// metrics to StatsD.
func (s *Server) StatHandle(p *pat.Pattern, handler handlerFunc, name string) {
	s.HandleFunc(p, func(w http.ResponseWriter, req *http.Request) {
		start := time.Now().UTC()

		handler(w, req)

		duration := time.Since(start)

		go func() {
			statName := fmt.Sprintf("health.%s", name)
			contextLogger := log.WithFields(log.Fields{
				"stat": statName,
			})

			err := s.statter.Inc(statName, 1, 0.1)
			if err != nil {
				contextLogger.WithError(err).Error("Failed to increment StatsD count type")
			}

			err = s.statter.TimingDuration(statName, duration, 0.1)
			if err != nil {
				contextLogger.WithError(err).Error("Failed to submit StatsD timing type")
			}
		}()
	})
}
