package middleware

import (
	"net/http"
	"time"

	log "github.com/Sirupsen/logrus"
)

const logFormat = "%s - [%s] \"%s %s %s\" %d %d - %dms"

type loggedResponseWriter struct {
	http.ResponseWriter
	status int
	data   []byte
}

// AccessLogger is an HTTP middleware that logs the request.
func AccessLogger(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, r *http.Request) {
		start := time.Now().UTC()
		logged := &loggedResponseWriter{
			ResponseWriter: w,
		}

		inner.ServeHTTP(logged, r)

		log.Infof(
			logFormat,
			r.RemoteAddr,
			start.Format("02/Jan/2006 03:04:05"),
			r.Method,
			r.RequestURI,
			r.Proto,
			logged.status,
			len(logged.data),
			time.Since(start)/time.Millisecond,
		)
	}

	return http.HandlerFunc(middleware)
}

func (l *loggedResponseWriter) Write(data []byte) (int, error) {
	l.data = data
	return l.ResponseWriter.Write(data)
}

func (l *loggedResponseWriter) WriteHeader(status int) {
	l.status = status
	l.ResponseWriter.WriteHeader(status)
}
