package middleware

import (
	"encoding/json"
	"net/http"

	log "github.com/Sirupsen/logrus"
)

// PanicRecoverer is an HTTP middleware that recovers from a panic,
// logs the panic, writes http.StatusInternalServerError as JSON.
func PanicRecoverer(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, r *http.Request) {
		defer func() {
			if err := recover(); err != nil {
				log.Errorf("[PANIC RECOVERY] %+v", err)

				status := http.StatusInternalServerError
				msg := http.StatusText(status)
				response := map[string]interface{}{
					"errors": []string{msg},
					"status": status,
				}

				w.Header().Set("Content-Type", "application/json")
				w.WriteHeader(status)

				err := json.NewEncoder(w).Encode(response)
				if err != nil {
					http.Error(w, msg, status)
				}
			}
		}()

		inner.ServeHTTP(w, r)
	}

	return http.HandlerFunc(middleware)
}
