package jobs

import (
	"context"
	"fmt"

	"code.justin.tv/cb/oracle/internal/clients"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/internal/clients/sns"
	"code.justin.tv/cb/oracle/internal/workerpool"
	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

type GetSubscribersFromDynamo struct {
	Event            *db.Event
	CacheKey         string
	Clients          *clients.Clients
	SNSQueue         chan<- workerpool.Job
	NotificationType sns.NotificationType
	GameName         string
}

func (job *GetSubscribersFromDynamo) Process() error {
	event := job.Event

	cachedEvent, err := job.Clients.CronCache.GetNotificationDelivery(job.CacheKey, event.ID)
	if err != nil {
		return err
	}

	if cachedEvent != nil {
		fmt.Println("hit cache event id ", event.ID)
		return nil
	}

	// continuously process dynamo query if paginating
	err = job.paginateDynamo(event)
	if err != nil {
		return err
	}

	err = job.Clients.CronCache.CacheNotificationDelivery(job.CacheKey, event.ID)
	if err != nil {
		return err
	}

	return nil
}

func (job *GetSubscribersFromDynamo) paginateDynamo(e *db.Event) error {
	var lastEvaluatedKey map[string]*dynamodb.AttributeValue
	for {
		lastEvaluatedKey, err := job.processDynamoQuery(e, lastEvaluatedKey)

		if err != nil {
			return err
		}
		if lastEvaluatedKey == nil {
			break
		}
	}
	return nil
}

// process dynamoquery gets the subscriptions from dynamo, returning the last key to paginate
func (job *GetSubscribersFromDynamo) processDynamoQuery(e *db.Event, lastEvaluatedKey map[string]*dynamodb.AttributeValue) (map[string]*dynamodb.AttributeValue, error) {
	subscriptions, lastEvaluatedKey, err := job.Clients.DynamoDB.GetUserNotificationsForEvent(context.Background(), e.ID, lastEvaluatedKey)
	if err != nil {
		return nil, err
	}

	for _, s := range subscriptions {
		cachedEvent, err := job.Clients.CronCache.GetUserNotificationDelivery(job.CacheKey, s.EventID, s.UserID)
		if err != nil {
			log.WithError(err).Error("processing dynamo query: cache failed")
			continue
		}

		if cachedEvent != nil {
			fmt.Println("hit cache event id ", s.EventID)
			continue
		}

		n := &SendNotificationSNS{
			UserID:           s.UserID,
			Event:            e,
			GameName:         job.GameName,
			Clients:          job.Clients,
			NotificationType: job.NotificationType,
		}
		job.SNSQueue <- n

		err = job.Clients.CronCache.CacheUserNotificationDelivery(job.CacheKey, s.EventID, s.UserID)
		if err != nil {
			log.WithError(err).Errorf("notifications dynamo job: failed to cache user %s and event %s notification", s.EventID, s.UserID)
			return nil, err
		}
		fmt.Println("added notification for event ID ", job.Event.ID, " and user id ", s.UserID)
	}

	return lastEvaluatedKey, nil
}
