package notifications

import (
	"context"
	"fmt"

	log "github.com/Sirupsen/logrus"

	"code.justin.tv/cb/oracle/internal/clients"
	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/internal/clients/sns"
	"code.justin.tv/cb/oracle/internal/notifications/jobs"
	"code.justin.tv/cb/oracle/internal/workerpool"
	"github.com/cactus/go-statsd-client/statsd"
)

type NotificationHandler struct {
	Clients *clients.Clients
	Stats   statsd.Statter
}

const (
	snsQueueSize  = 200
	snsNumWorkers = 50

	dynamoNumWorkers = 50
)

func (n *NotificationHandler) NotifySubscribersForEvents(eventsFetchFn EventsFetchFn, notificationType sns.NotificationType, cacheKey string) {
	defer func() {
		if r := recover(); r != nil {
			log.WithField("recover", r).Errorf("notifications: recovered from panic for  '%s'", cacheKey)
		}
	}()

	events := eventsFetchFn()

	dynamoJobsQueue := make(chan workerpool.Job)
	snsJobsQueue := make(chan workerpool.Job, snsQueueSize)

	go n.addEventsToDynamoJobsQueue(events, notificationType, dynamoJobsQueue, snsJobsQueue, cacheKey)

	err := n.processDynamoJobsQueue(dynamoJobsQueue, snsJobsQueue)
	if err != nil {
		log.WithError(err).Errorf("notifications: failed to process dynamo jobs queue for %s", cacheKey)
	}

	err = n.processSNSJobsQueue(snsJobsQueue, cacheKey)
	if err != nil {
		log.WithError(err).Errorf("notifications: failed to process sns jobs queue for %s", cacheKey)
	}
}

func (n *NotificationHandler) addEventsToDynamoJobsQueue(events []*db.Event, notificationType sns.NotificationType, dynamoJobsQueue chan<- workerpool.Job, snsJobsQueue chan<- workerpool.Job, cacheKey string) {
	for _, e := range events {
		game, err := n.Clients.Discovery.GetGameByID(context.Background(), e.GameID)
		if err != nil {
			log.WithError(err).Errorf("notifications: error getting game name for event id %d", e.ID)
		}

		var gameName string
		if game != nil {
			gameName = game.Name
		}

		dynamoJobsQueue <- &jobs.GetSubscribersFromDynamo{
			Event:            e,
			CacheKey:         cacheKey,
			Clients:          n.Clients,
			SNSQueue:         snsJobsQueue,
			GameName:         gameName,
			NotificationType: notificationType,
		}

		statName := "dynamo_event_read"
		err = n.Stats.Inc(statName, 1, 0.1)
		if err != nil {
			log.WithError(err).Error(fmt.Sprintf("Failed to increment stat for %s", statName))
		}
	}
	close(dynamoJobsQueue)
}

func (n *NotificationHandler) processDynamoJobsQueue(dynamoJobsQueue <-chan workerpool.Job, snsQueue chan<- workerpool.Job) error {
	dynamoWorkerPool := &workerpool.WorkerPool{
		JobQueue:   dynamoJobsQueue,
		NumWorkers: dynamoNumWorkers,
		Stats:      n.Stats,
		StatName:   "dynamo_event_subscribers_read",
	}

	wg, err := dynamoWorkerPool.Run()
	if err != nil {
		return err
	}
	go func() {
		wg.Wait()
		close(snsQueue)
	}()
	return nil
}

func (n *NotificationHandler) processSNSJobsQueue(snsJobsQueue <-chan workerpool.Job, cacheKey string) error {
	snsWorkerPool := &workerpool.WorkerPool{
		JobQueue:   snsJobsQueue,
		NumWorkers: snsNumWorkers,
		Stats:      n.Stats,
		StatName:   "sns_topic_send",
	}

	wg, err := snsWorkerPool.Run()
	if err != nil {
		return err
	}

	go func() {
		wg.Wait()
		fmt.Println("all done for", cacheKey)
	}()
	return nil
}
